﻿using System;
using System.Net.Mail;
using System.Text;
using System.Text.RegularExpressions;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;

namespace Demo.Classes
{
    public abstract class BaseErrorPage : Page
    {
        #region Fields

        protected Literal BrokenUrl;
        protected Literal CrashReportBody;
        protected Literal CrashReportTitle;
        protected Panel ErrorReportPanel;
        protected NextStepsLinkList NextSteps;
        protected HiddenField ReportBody;
        protected HiddenField ReportTitle;
        protected HiddenField StatusCode;
        protected LinkButton SubmitQuickErrorReportButton;
        private Int32 _httpStatusCode;
        
        #endregion

        #region Methods (initialization and loading)

        protected override void OnInit(EventArgs e)
        {
            base.OnInit(e);

            EnableViewStateMac = false;
            SubmitQuickErrorReportButton.Click += SubmitQuickErrorReportButtonOnClick;
        }

        protected override void OnLoad(EventArgs e)
        {
            // This is the base page from which my custom error pages inherit. If an unhandled exception occurs here then 
            // my custom error page is not displayed to the user. Instead, the user sees the default ASP.NET Server Error 
            // page (the "Yellow Screen of Death"). In effect, if an unhandled exception occurs in the error page, then all 
            // my custom error handling is overridden, and the original exception details are lost.

            // throw new DemoException("What happens to an unhandled exception in the custom error page?");

            base.OnLoad(e);
            
            _httpStatusCode = Int32.Parse(StatusCode.Value);
            Response.StatusCode = _httpStatusCode;
            
            if (!IsPostBack)
            {
                Form.Action = Request.Url.PathAndQuery;

                if (_httpStatusCode == 404)
                {
                    CrashReportTitle.Text = "Page Not Found";
                    CrashReportBody.Text = String.Empty;

                    ReportTitle.Value = "Page Not Found";
                    ReportBody.Value = String.Empty;
                }
                else
                {
                    ReportTitle.Value = "Unhandled Exception";
                    ReportBody.Value = String.Empty;

                    var report = HttpContext.Current.Cache[ApplicationErrorModule.Settings.Names.CrashReportKey] as CrashReport;
                    if (report == null)
                    {
                        Exception ex = HttpContext.Current.Server.GetLastError() ?? new HttpUnhandledException("Server Error");
                        report = new CrashReport(ex);
                    }

                    CrashReportTitle.Text = report.Title;
                    CrashReportBody.Text = report.Body;

                    ReportTitle.Value = report.Title;
                    ReportBody.Value = report.Body;

                    NextSteps.EnableTechnicalInformation = true;
                }

                BrokenUrl.Text = GetBrokenUrl();
            }
            else
            {
                // If this is a postback then we're OK -- the user is submitting a quick question from the error page.
                Response.StatusCode = 200;
            }
        }

        #endregion

        #region Methods (event handling)

        private void SubmitQuickErrorReportButtonOnClick(Object sender, EventArgs eventArgs)
        {
            SendQuickErrorReport(BrokenUrl.Text, GetUserName(), Request.Browser, ReportTitle.Value, ReportBody.Value);

            RegisterThankYouScript();

            ErrorReportPanel.Visible = false;

            Response.StatusCode = 200;
        }

        #endregion

        #region Methods (helpers)

        private String GetBrokenUrl()
        {
            if (Request.QueryString["aspxerrorpath"] != null)
                return Server.UrlDecode(Request.QueryString["aspxerrorpath"]);

            String query = Server.UrlDecode(Request.QueryString.ToString());
            if (!String.IsNullOrEmpty(query))
            {
                const String pattern = @"^(404|500);(.+):80(.+)$";
                Match match = Regex.Match(query, pattern);
                if (match.Success)
                    return match.Groups[3].Value;
            }

            return Request.RawUrl;
        }

        private String GetUserName()
        {
            String user = "Anonymous";
            if (Request.IsAuthenticated && Page.User != null)
                user = Page.User.Identity.Name;
            return user;
        }

        private void RegisterThankYouScript()
        {
            var script = new StringBuilder();
            script.Append(@"<script language='javascript'>");
            script.AppendFormat(@"alert('Thanks for reporting this {0}.');<", _httpStatusCode == 404 ? "missing page" : "exception notice");
            script.Append(@"/script>");

            if (!ClientScript.IsStartupScriptRegistered("JSScript"))
                ClientScript.RegisterStartupScript(GetType(), "JSScript", script.ToString());
        }

        private static void SendQuickErrorReport(String url, String user, HttpBrowserCapabilities browser, String title, String body)
        {
            var message = new MailMessage();
            message.To.Add(ApplicationErrorModule.Settings.CrashReportEmailAddress);
            message.Subject = "Quick Error Report: " + title;

            var html = new StringBuilder();

            html.Append("<style>");
            html.Append("*, body, p, table { font-family: calibri; font-size: 11pt; }");
            html.Append("table { border-collapse: collapse; }");
            html.Append("table tr th { text-align: left; padding-top: 10px; padding-bottom: 10px; }");
            html.Append("table tr td { border: 1px solid #cccccc; vertical-align: top; padding: 5px; }");
            html.Append("table tr td.label { font-weight: bold; color: rgb(38, 95, 159); }");
            html.Append("</style>");

            if (url != null)
                html.AppendFormat("Url: {0}<br />", url);

            if (browser != null)
                html.AppendFormat("Browser: {0} {1} {2}<br />", browser.Platform, browser.Browser, browser.Version);

            if (user != null)
                html.AppendFormat("User: {0}<br />", user);

            html.Append("<br />");

            html.Append(body);

            message.Body = html.ToString();
            message.IsBodyHtml = true;

            var smtp = new SmtpClient();
            smtp.Send(message);
        }

        #endregion
    }
}