﻿using System;
using System.Linq;
using System.Collections.Generic;
using SolutionArch.BusinessObject;
using System.Reflection;


namespace SolutionArch.Collection.Entity
{
    public static class Utility 
    {
        class Expression
        {
            public PropertyInfo     PropertyInfo;
            public string           ExpressionPropertyName;
            public string           ExpressionPropertyVal;
        }

        /// <summary>
        /// Queries datasource based on expression
        /// </summary>
        /// <typeparam name="T">Generic data source</typeparam>
        /// <param name="EntityCollection">Data source - Collection</param>
        /// <param name="FindExpression">Find expression  as Property = Value </param>
        /// <returns></returns>
        public static EntityCollection<T> FindByExpression<T>(this EntityCollection<T> EntityCollection, string FindExpression)
        {
            Expression          objExpression   = GetPropertyInfo(EntityCollection, FindExpression, '=');
            // Find all matches -- Lamda Expression
            IEnumerable<T>      objEntityList   = EntityCollection.FindAll(objC => objExpression.PropertyInfo.GetValue(objC, null).ToString()
                                                                              == objExpression.ExpressionPropertyVal);
            EntityCollection<T> objEntities     = new EntityCollection<T>();
            // Create a new entity collection with mathced item
            foreach (var objEntity in objEntityList) objEntities.Add(objEntity);
            return objEntities;
        }

        /// <summary>
        /// Remove element from the collection based on the query expression
        /// </summary>
        /// <typeparam name="T">Generic data source</typeparam>
        /// <param name="EntityCollection">Data source - Collection</param>
        /// <param name="FindExpression">Find expression  as Property = Value</param>
        /// <returns></returns>
        public static EntityCollection<T> DeleteByExpression<T>(this EntityCollection<T> EntityCollection, string FindExpression)
        {
            Expression      objExpression    = GetPropertyInfo(EntityCollection, FindExpression, '=');
            // Find all matches -- Lamda Expression
            IEnumerable<T>  objEntityList    = EntityCollection.FindAll(objC => objExpression.PropertyInfo.GetValue(objC, null).ToString()
                                                                        == objExpression.ExpressionPropertyVal);
            // Remove the matches from entity list
            foreach (var objEntity in objEntityList) EntityCollection.Remove(objEntity);
            return EntityCollection;
        }

        /// <summary>
        /// Sort datasource based on expression
        /// </summary>
        /// <typeparam name="T">Generic data source</typeparam>
        /// <param name="EntityCollection">Data source - Collection</param>
        /// <param name="SortExpression">Sort expression  as Property Desc , Default sort order is ASC</param>
        /// <returns></returns>
        public static EntityCollection<T> SortByExpression<T>(this EntityCollection<T> EntityCollection, string SortExpression)
        {
            bool            blnDescending = false;
            IEnumerable<T>  objEntityList;
            Expression      objExpression = GetPropertyInfo(EntityCollection, SortExpression, ' ');

            blnDescending = SortExpression.ToUpper().Contains("DESC");
            // Sort in Desc order
            if (blnDescending) objEntityList = EntityCollection.OrderByDescending(objX => objExpression.PropertyInfo.GetValue(objX, null));
            // Sort in ASC order
            else objEntityList = EntityCollection.OrderBy(objX => objExpression.PropertyInfo.GetValue(objX, null));

            EntityCollection<T> objEntities = new EntityCollection<T>();
            // Create a new entity collection with sorted items
            foreach (var objEntity in objEntityList)  objEntities.Add(objEntity);
            return objEntities;
        }

        /// <summary>
        /// Populate property info
        /// </summary>
        /// <typeparam name="T">Generic data source</typeparam>
        /// <param name="EntityCollection">Data source - Collection</param>
        /// <param name="strExpression">Expression</param>
        /// <param name="chSplitChar">Split Char</param>
        /// <returns></returns>
        private static Expression GetPropertyInfo<T>(EntityCollection<T> EntityCollection, string strExpression, char chSplitChar)
        {
            // Expression format should be like 
            // Property Desc  - for sort operation
            // Property = value - for Find and delete operation
            // parts[0] will have property name
            // Parts[1] will have sort order or value
            string[] strParts                       = strExpression.Trim().Split(chSplitChar);

            // Property details using reflection
            PropertyInfo objPropInfo                = typeof(T).GetProperty(strParts[0]);

            // Incase of invalid property throw an exception
            if (objPropInfo == null) throw new Exception("Invalid '" + strParts[0] + "' in + " + typeof(T).Name + "'");
            
            Expression objExpression                = new Expression();
            objExpression.PropertyInfo              = objPropInfo;
            objExpression.ExpressionPropertyName    = strParts[0];
            objExpression.ExpressionPropertyVal     = strParts.Length > 1 ? strParts[1] : string.Empty;

            return objExpression;
        }

    }
}
