/**
 * jquery.checkin.js v1.0.0
 *
 * Licensed under the MIT license.
 * http://www.opensource.org/licenses/mit-license.php
 *
 * Copyright 2013, Francesco Carrella
 * http://www.francescocarrella.com
 */
;
(function($, window, undefined) {
	'use strict';
	$.CheckIn = function(options, element) {
		this.$el = $(element);
		this._init(options);
	};



	/*****************************************************************
	 *** Globals default options
	 *****************************************************************/
	$.CheckIn.defaults = {

		/**
		 * @cfg {String[]} weeks
		 * Array containing localized week days names.
		 */
		weeks: ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'],


		/**
		 * @cfg {String[]} weekabbrs
		 * Array containing localized week days short names.
		 */
		weekabbrs: ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'],


		/**
		 * @cfg {String[]} months
		 * Array containing localized months names.
		 */
		months: ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'],


		/**
		 * @cfg {String[]} monthabbrs
		 * Array containing localized months short names.
		 */
		monthabbrs: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'],


		/**
		 * @cfg {Boolean} displayWeekAbbr
		 * Choose between values in options.weeks or options.weekabbrs
		 */
		displayWeekAbbr: false,


		/**
		 * @cfg {Boolean} displayMonthAbbr
		 * Choose between values in options.months or options.monthabbrs
		 * */
		displayMonthAbbr: false,


		/**
		 * @cfg {Boolean} weekStartDay
		 * Left most day in the calendar
		 * Example:
		 *     0 - Sunday
		 *     1 - Monday
		 *     ...
		 *     6 - Saturday
		 */
		weekStartDay: 1,


		/**
		 * @cfg {Boolean} allowNonContiguousSelection
		 * By default deny non contiguous time range selections.
		 */
		allowNonContiguousSelection: false,


		/**
		 * @cfg {Boolean} allowPartialsSelection
		 * By default allow partial selection; If user finish selection on
		 * disabled day, will be selected only valid days
		 */
		allowPartialsSelection: true


		/**
		 * @cfg {Number} month
		 * Initialize calendar with this month (1-12). Default is today's month
		 */


		/**
		 * @cfg {Number} year
		 * Initialize calendar with this year (2014, 2015, 2016, ...). Default is today's year
		 */


		/**
		 * @cfg {Object} daysData
		 * A diccionary of data objects associated with any day or any days range and that can contains
		 * all application-needed data, as price associated to specific day, available places, etc.
		 *
		 * The key (property name) of each dictionary item should be a string representing a date or a date range.
		 * Formats actually accepted are:
		 *     "YYYY-MM-DD"               - For a single day data
		 *     "YYYY-MM-DD:YYYY-MM-DD"    - For days range data
		 *
		 * The day(s)-related data always will be returned in context of range selection events,
		 * or eventually, you can ask for that through getDayData(date) method.
		 *
		 * Optionally, the day data config can contains the 'disabled' bool property that
		 * can automagically unable time-range selection for this day.
		 *
		 * Example:
		 *
		 *      {
		 *           // Single day
		 *           'YYYY-MM-DD' : {
		 *               disabled: true,
		 *               myValue: '...'
		 *           },
		 *
		 *           // Data range
		 *           'YYYY-MM-DD:YYYY-MM-DD' : {
		 *               disabled: false,
		 *               myOtherValue: '...'
		 *           },
		 *
		 *           ...
		 *       }
		 *
		 */


	};



	$.CheckIn.prototype = {


		/*****************************************************************
		 *** Init methods
		 *****************************************************************/

		/**
		 * Global initializator method; it will be called by jQuery plugin wrapper (see at the end of this file).
		 * @param {object} options Instance configuration options; it will be merged with $.CheckIn.defaults to get defaults values
		 */
		_init: function(options) {
			var self = this;
			self.options = $.extend(true, {}, $.CheckIn.defaults, options);
			if (self.options.disableBefore && typeof self.options.disableBefore === 'string') {
				self.options.disableBefore = self._parseDateString(self.options.disableBefore);
			}
			self.today = new Date();
			self.month = (isNaN(self.options.month) || self.options.month === null) ? self.today.getMonth() : self.options.month - 1;
			self.year = (isNaN(self.options.year) || self.options.year === null) ? self.today.getFullYear() : self.options.year;
			self.selection = {};
			self._initDaysData();
			self._generateTemplate();
			self._initEvents();
		},

		/**
		 * Transform passed days data structure to an internal one
		 */
		_initDaysData: function() {
			var self = this;
			self.daysData = [];
			if (self.options.daysData) {
				$.each(self.options.daysData, function(i, dayData) {
					var dataRange = i.split(':');
					self.daysData.push({
						start: self._parseDateString(dataRange[0]),
						end: self._parseDateString(dataRange[1] || dataRange[0]),
						data: dayData
					});
				});
			}
		},

		/**
		 * Init user interaction events.
		 * If jQuery.hammer plugin is present, it will be used to manage touch events
		 *  - more info: http://eightmedia.github.io/hammer.js
		 */
		_initEvents: function() {
			var self = this;

			if ($.fn.hammer) {
				self.$el.hammer();
				self.$el.on({
					tap: function(e) {
						var $cell = $(document.elementFromPoint(e.gesture.center.pageX - window.scrollX, e.gesture.center.pageY - window.scrollY)).closest('.checkin-day-cell')
						if ($cell.length) {
							self._onDayClick.call(self, $cell);
						}
					},
					dragstart: function(e) {
						var $cell = $(document.elementFromPoint(e.gesture.center.pageX - window.scrollX, e.gesture.center.pageY - window.scrollY)).closest('.checkin-day-cell'),
							date = $cell.data('checkin-day-date');
						if ($cell.length) {
							self._startSelection.call(self, date);
						}
					},
					drag: function(e) {
						var $cell = $(document.elementFromPoint(e.gesture.center.pageX - window.scrollX, e.gesture.center.pageY - window.scrollY)).closest('.checkin-day-cell');
						if ($cell.length) {
							self._onDayHover.call(self, $cell);
						}
					},
					dragend: function(e) {
						var $cell = $(document.elementFromPoint(e.gesture.center.pageX - window.scrollX, e.gesture.center.pageY - window.scrollY)).closest('.checkin-day-cell');
						if (!$cell.length) {
							$cell = self.$daysCells.filter('.checkin-day-highlight').last();
						}
						var date = $cell.data('checkin-day-date');
						self._stopSelection.call(self, date);
					}
				});

			} else {
				self.$el.on({
					click: function() {
						var $cell = $(this).closest('.checkin-day-cell');
						self._onDayClick.call(self, $cell);
					}
				}, '.checkin-day-cell, .checkin-day-cell *');
			}
			self.$el.on({
				hover: function() {
					var $cell = $(this).closest('.checkin-day-cell');
					self._onDayHover.call(self, $cell);
				}
			}, '.checkin-day-cell, .checkin-day-cell *');
		},



		/*****************************************************************
		 *** Renderers methods
		 *****************************************************************/

		/**
		 * General render method that create the global HTML structure.
		 * It will be called on each refresh, month change & days data change.
		 * @param  {Function} callback Method that will be called at the end of template creation
		 */
		_generateTemplate: function(callback) {
			var self = this,
				head = self._getHead(),
				body = self._getBody(),
				rowClass;
			switch (self.rowTotal) {
				case 4:
					rowClass = 'checkin-four-rows';
					break;
				case 5:
					rowClass = 'checkin-five-rows';
					break;
				case 6:
					rowClass = 'checkin-six-rows';
					break;
			}
			self.$cal = $('<div>', {
				'class': 'checkin-calendar ' + rowClass
			}).append(head, body);

			self.$el.find('div.checkin-calendar').remove().end().append(self.$cal);
			self.$daysCells = $('.checkin-day-cell', self.$el);

			if (self.selection.startDate && self.selection.endDate) {
				self._drawSelection(self.selection.startDate, self.selection.endDate);
			}

			if (callback && typeof callback == 'function') {
				callback.call();
			}

			self.$el.trigger('checkin.rendered');
		},

		/**
		 * Render weekdays header.
		 * @return {jQuery object} Header object that contains DOM header structure
		 */
		_getHead: function() {
			var self = this,
				$head = $('<div>', {
					'class': 'checkin-head'
				});
			for (var i = 0; i <= 6; i++) {
				var pos = i + self.options.weekStartDay,
					j = pos > 6 ? pos - 6 - 1 : pos;
				$('<div>', {
					html: self.options.displayWeekAbbr ? self.options.weekabbrs[j] : self.options.weeks[j]
				}).appendTo($head);
			}
			return $head;
		},

		/**
		 * Render calendar body content, essentially week rows and day cells.
		 * @return {jQuery object} Calendar body object that contains DOM structure
		 */
		_getBody: function() {
			var self = this,
				firstMonthDay = new Date(self.year, self.month, 1),
				lastMonthDay = new Date(self.year, self.month + 1, 0),
				monthLength = lastMonthDay.getDate(),
				day = 1,
				$body = $('<div>', {
					'class': 'checkin-body'
				});

			// day of the week
			self.startingDay = firstMonthDay.getDay();

			var startingDayPosition = self.startingDay;
			if (self.options.weekStartDay) {
				startingDayPosition = startingDayPosition - self.options.weekStartDay;
				if (startingDayPosition < 0) {
					startingDayPosition = 6 + startingDayPosition + 1;
				}
			}

			// this loop draw weeks (rows)
			for (var i = 0; i < 7; i++) {

				var $weekRow = $('<div>', {
					'class': 'checkin-row'
				}).appendTo($body);

				// this loop draw weekdays (cells)
				for (var j = 0; j <= 6; j++) {

					var $dayCell = $('<div>').appendTo($weekRow),
						today = self.month === self.today.getMonth() && self.year === self.today.getFullYear() && day === self.today.getDate();

					if (day <= monthLength && (i > 0 || j >= startingDayPosition)) {
						var date = new Date(self.year, self.month, day, 0, 0, 0, 0),
							dayData = self.getDayData(date),
							$dayNumber = $('<span>', {
								'class': 'checkin-date',
								html: day
							}).appendTo($dayCell);
						$dayCell
							.data('checkin-day-date', date)
							.addClass('checkin-day-cell')
							.toggleClass('checkin-day-disabled', (dayData.disabled !== undefined && dayData.disabled))
							.toggleClass('checkin-today', today);
						self.$el.trigger('checkin.dayrendered', [$dayCell, dayData]);
						++day;
					}
				}

				// stop making rows if we've run out of days
				if (day > monthLength) {
					self.rowTotal = i + 1;
					break;
				}
			}
			return $body;
		},

		/**
		 * Draw a days selection applying a css class to each day cell included in selection.
		 * Additionally this method can apply different css class to context "unselectables" days
		 * @param  {Date} dateStart               To select range start data
		 * @param  {Date} dateEnd                 To select range end data
		 * @param  {String} selectedCssClass      Css class that will be applied to selected day cells
		 * @param  {String} unselectableCssClass  If passed this class will be applied to unselectables day cells
		 * @return {jQuery object}                All calendar cells as jQuery object
		 */
		_drawSelection: function(dateStart, dateEnd, selectedCssClass, unselectableCssClass) {
			var self = this,
				isUnselectable = false,
				isInverseSelection = false;

			// Invert selection
			if (dateStart > dateEnd) {
				dateEnd = [dateStart, dateStart = dateEnd][0];
				isInverseSelection = true;
			}

			self._clearSelection();

			if (typeof(selectedCssClass) !== 'string') {
				selectedCssClass = 'checkin-day-selected';
			}
			return $.each(self.$daysCells, function() {
				var $cell = $(this),
					date = $cell.data('checkin-day-date'),
					dayData = self.getDayData(date);

				if (!self.options.allowNonContiguousSelection && unselectableCssClass) {
					if (date >= dateStart && dayData.disabled) {
						isUnselectable = true;
					}
					$cell.toggleClass(unselectableCssClass, isUnselectable && !isInverseSelection);
				}

				$cell.toggleClass(selectedCssClass, date === dateStart || (date >= dateStart && date <= dateEnd));

			});
		},

		/**
		 * Remove all css classes from days cells.
		 */
		_clearSelection: function() {
			self.$daysCells.removeClass('checkin-day-selected checkin-day-highlight checkin-day-unselectable');
		},



		/*****************************************************************
		 *** Selection related methods
		 *****************************************************************/

		/**
		 * Start a days range selection.
		 * @param  {Date} date    From date
		 */
		_startSelection: function(date) {
			var self = this,
				dayData = self.getDayData(date);

			if (!dayData.disabled) {
				self.resetSelection();
				self.selection.startDate = date;
				self._drawSelection(date, date, null, 'checkin-day-unselectable');
				self.$cal.addClass('selecting');
				self.$el.trigger('checkin.selection.start', self.selection);
			}
		},

		/**
		 * Finalize a days range selection.
		 * @param  {Date} date    To date
		 */
		_stopSelection: function(date) {
			var self = this,
				dayData = self.getDayData(date),
				selectedRanges = self._getEnabledDaysRanges(self.selection.startDate, date),
				isPartialSelection = false,
				isInvertedSelection = false;


			// User inverted selection, started from the end date and finished to start date
			if (date < self.selection.startDate) {
				date = [self.selection.startDate, self.selection.startDate = date][0];
				dayData = self.getDayData(date);
				selectedRanges = self._getEnabledDaysRanges(self.selection.startDate, date);
				isInvertedSelection = true;
			}


			// Check if selection include disabled days 
			if (self._rangeContainsDisabledDays(self.selection.startDate, date)) {

				self.$el.trigger('checkin.error.includedisabledday');

				if (!self.options.allowNonContiguousSelection) {
					var rangeToSelect = isInvertedSelection ? selectedRanges.length - 1 : 0,
						selectedRange = selectedRanges[rangeToSelect];
					self.selection.startDate = selectedRange.start;
					date = selectedRange.end;
					self.$el.trigger('checkin.error.noncontiguousselection', selectedRange);
				}

				if (!self.options.allowPartialsSelection) {
					self.resetSelection();
					return;
				}
			}

			self.selection.endDate = date;
			self._drawSelection(self.selection.startDate, self.selection.endDate);
			self.$cal.removeClass('selecting');

			self.$el.trigger('checkin.selection.finish', [selectedRanges.length > 1 ? selectedRanges : selectedRanges[0]]);

		},



		/*****************************************************************
		 *** User actions days cells listeners
		 *****************************************************************/

		/**
		 * Handler for click event; it will start or stop selection.
		 * @param  {jQuery object} $cell Clicked cell
		 */
		_onDayClick: function($cell) {
			var self = this,
				date = $cell.data('checkin-day-date');
			if (self.selection.startDate && !self.selection.endDate) {
				self._stopSelection(date);
			} else {
				self._startSelection(date);
			}
		},

		/**
		 * Handler for mouse move event; it will draw a selection to highlight current selecting days.
		 * @param  {jQuery object} $cell Hover cell
		 */
		_onDayHover: function($cell) {
			var self = this,
				date = $cell.data('checkin-day-date');
			if (self.selection.startDate && !self.selection.endDate) {
				self._drawSelection(self.selection.startDate, date, 'checkin-day-highlight', 'checkin-day-unselectable');
			}
		},



		/*****************************************************************
		 *** Range utils
		 *****************************************************************/

		/**
		 * Return a data structure representing a range of NOT disabled days (or an array of that).
		 * See public getSelection method docs to know more about ranges structure.
		 * @param  {Date} start        Period start date
		 * @param  {Date} end          Period end date
		 * @return {Object[] / Object} Object representing range structure (or, in case of multiples ranges, an array of its)
		 */
		_getEnabledDaysRanges: function(start, end) {
			var ranges = [],
				currentRange,
				previousRange,
				previousDay,
				previousDayData;

			for (var d = new Date(start.getTime()); d <= end; d.setDate(d.getDate() + 1)) {
				var dayData = self.getDayData(d);

				// Create a new range if it's needed
				if (!currentRange || (dayData.disabled && !previousDayData.disabled)) {
					if (currentRange) {
						previousRange = currentRange;
						previousRange.end = previousDay;
					}
					currentRange = {};
					currentRange.daysData = [];
					ranges.push(currentRange);
				}

				// Add day to current range
				if (!dayData.disabled) {
					if (!currentRange.start) {
						currentRange.start = new Date(d);
					}
					currentRange.daysData.push(dayData);
				}

				// Store current day values for further use
				previousDay = new Date(d);
				previousDayData = dayData;
			}
			if (currentRange) {
				currentRange.end = new Date(end);
			}
			return ranges;
		},

		/**
		 * To know if a certain period contains any disabled day.
		 * @param  {Date} start   Period start date
		 * @param  {Date} end     Period end date
		 * @return {Boolean}      Indicates if passed period contains disabled days
		 */
		_rangeContainsDisabledDays: function(start, end) {
			var containsDisabled = false;
			for (var d = new Date(start.getTime()); d <= end; d.setDate(d.getDate() + 1)) {
				var dayData = self.getDayData(d);
				if (dayData.disabled) {
					containsDisabled = true;
				}
				if (containsDisabled) {
					break;
				}
			}
			return containsDisabled;
		},



		/*****************************************************************
		 *** Utils methods
		 *****************************************************************/

		/**
		 * Parse date string to js native data object.
		 * @param  {String} date Dates string in format 'YYYY-MM-DD'
		 * @return {Date}        Native js Date object
		 */
		_parseDateString: function(date) {
			date = date.replace(/-/gi, '');
			var year = parseInt(date.substring(0, 4), 10),
				month = parseInt(date.substring(4, 6), 10),
				day = parseInt(date.substring(6, 8), 10);
			if ((month < 1) || (month > 12)) {
				return false;
			} else if ((day < 1) || (day > 31)) {
				return false;
			} else if (((month == 4) || (month == 6) || (month == 9) || (month == 11)) && (day > 30)) {
				return false;
			} else if ((month == 2) && (((year % 400) === 0) || ((year % 4) === 0)) && ((year % 100) !== 0) && (day > 29)) {
				return false;
			} else if ((month == 2) && ((year % 100) === 0) && (day > 29)) {
				return false;
			}
			return new Date(year, month - 1, day, 0, 0, 0, 0);
		},

		/**
		 * Format date to 'YYYY-MM-DD' string.
		 * @param  {Date} date
		 * @return {String}      'YYYY-MM-DD' formatted string
		 */
		_getDayString: function(date) {
			return date.getFullYear() + '-' + this._padNumber(date.getMonth() + 1) + '-' + this._padNumber(date.getDate());
		},

		/**
		 * Convert a number to 2 digit string.
		 * @param  {Number} number
		 * @return {String}         2 digit string
		 */
		_padNumber: function(number) {
			return (number < 10 ? '0' : '') + number;
		},



		/*****************************************************************
		 *** Movements utils
		 *****************************************************************/

		/**
		 * Render the next or prev period (month or year).
		 * @param  {String]}   period   Indicates the "kind of" period ('month'|'year')
		 * @param  {String]}   dir      Specify direction of movement ('next|'previous')
		 * @param  {Function}  callback
		 */
		_moveToPeriod: function(period, dir, callback) {
			var self = this;
			if (dir === 'previous') {
				if (period === 'month') {
					self.year = self.month > 0 ? self.year : --self.year;
					self.month = self.month > 0 ? --self.month : 11;
				} else if (period === 'year') {
					self.year = --self.year;
				}
			} else if (dir === 'next') {
				if (period === 'month') {
					self.year = self.month < 11 ? self.year : ++self.year;
					self.month = self.month < 11 ? ++self.month : 0;
				} else if (period === 'year') {
					self.year = ++self.year;
				}
			}
			self._generateTemplate(callback);
		},



		/*****************************************************************
		 *** Public exposed methods
		 *****************************************************************/

		/**
		 * Gets currently selected year.
		 * @return {Number} Year number (2013, 2014, 2015, ...)
		 */
		getYear: function() {
			return this.year;
		},

		/**
		 * Gets currently selected month.
		 * @return {Number} Month number (1...12)
		 */
		getMonth: function() {
			return this.month + 1;
		},

		/**
		 * Gets currently selected month name, depending by current applied language tag.
		 * @return {String} Month name
		 */
		getMonthName: function() {
			return this.options.displayMonthAbbr ? this.options.monthabbrs[this.month] : this.options.months[this.month];
		},

		/**
		 * Gets the cell's content div associated to a day of the current displayed month.
		 * @param  {Number} day     Day number of current month (1 ... [28|29|30|31])
		 * @return {jQuery object}  Contains the DOM object associated to cell representing passed day of current month
		 */
		getCell: function(day) {
			var self = this;
			var row = Math.floor((day + self.startingDay - self.options.weekStartDay) / 7),
				pos = day + self.startingDay - self.options.weekStartDay - (row * 7) - 1;
			return self.$cal.find('div.checkin-body').children('div.checkin-row').eq(row).children('div').eq(pos).children('div');
		},

		/**
		 * Return data associated to passed day; if day is not provided, return config for all days or day range that have one.
		 * @param  {Date}     day
		 * @return {Object | Array(Object)}   Dictionary containing day data (or an array of that)
		 */
		getDayData: function(day) {
			var self = this;

			if (!day) {
				return self.daysData;
			}

			var dayData = {};
			if (typeof day === 'string') {
				day = self._parseDateString(day);
			}
			$.each(self.daysData, function(i, dayObj) {
				if (day.getTime() >= dayObj.start.getTime() && day.getTime() <= dayObj.end.getTime()) {
					$.extend(dayData, dayObj.data);
				}
			});
			if (self.options.disableBefore && day <= self.options.disableBefore) {
				dayData.disabled = true;
			}
			if (self.options.disableAfter && day >= self.options.disableAfter) {
				dayData.disabled = true;
			}
			return dayData;
		},

		/**
		 * Set a new data for days. See "Globals default options" section.
		 * @param  {Object} data             Data object. Read "Globals default options" section for more info
		 * @param  {Boolean} removePrevious  Clean previous setted data
		 */
		setDaysData: function(data, removePrevious) {
			var self = this;
			console.log(self);
			if (removePrevious) {
				self.options.daysData = [];
			}
			data = data || {};
			$.extend(self.options.daysData, data);
			self._initDaysData();
			self._generateTemplate();
		},

		/**
		 * Return current selected ranges as a diccionary like:
		 * {
		 *      start: Date,
		 *      end: Date,
		 *      data: [{ ...first day data... }, { ...second day data... }, ...]
		 * }
		 * If multiples ranges are selected (according with allowNonContiguousSelection option),
		 * an array of ranges dictionaries will be returned.
		 * @return {Object | Array(Object)} [description]
		 */
		getSelection: function() {
			if (!self.selection || !self.selection.startDate || !self.selection.endDate) {
				return false;
			}
			var selectedRanges = self._getEnabledDaysRanges(self.selection.startDate, self.selection.endDate);
			return selectedRanges.length > 1 ? selectedRanges : selectedRanges[0];
		},

		/**
		 * Remove current selection.
		 */
		resetSelection: function() {
			self = this;
			self.selection = {};
			self._clearSelection();
		},

		/**
		 * Goes to today's month / year.
		 * @param  {Function} callback
		 */
		gotoNow: function(callback) {
			var self = this;
			self.month = self.today.getMonth();
			self.year = self.today.getFullYear();
			self._generateTemplate(callback);
		},

		/**
		 * Goes to specific mont / year.
		 * @param  {Number}   month
		 * @param  {Number}   year
		 * @param  {Function} callback
		 */
		goto: function(month, year, callback) {
			var self = this;
			self.month = month || self.month;
			self.year = year || self.year;
			self._generateTemplate(callback);
		},

		/**
		 * Goes to previous month.
		 * @param  {Function} callback
		 */
		gotoPreviousMonth: function(callback) {
			var self = this;
			self._moveToPeriod('month', 'previous', callback);
		},

		/**
		 * Goes to previous year.
		 * @param  {Function} callback
		 */
		gotoPreviousYear: function(callback) {
			var self = this;
			self._moveToPeriod('year', 'previous', callback);
		},

		/**
		 * Goes to next month.
		 * @param  {Function} callback
		 */
		gotoNextMonth: function(callback) {
			var self = this;
			self._moveToPeriod('month', 'next', callback);
		},

		/**
		 * Goes to next year.
		 * @param  {Function} callback
		 */
		gotoNextYear: function(callback) {
			var self = this;
			self._moveToPeriod('year', 'next', callback);
		}
	};

	$.fn.checkin = function(options) {
		var args = Array.prototype.slice.call(arguments, 1),
			dataToReturn;
		this.each(function() {
			var instance = $.data(this, 'checkin');
			if (typeof options === 'string') {
				if (!instance) {
					logError("cannot call methods on checkin prior to initialization; " +
						"attempted to call method '" + options + "'");
					return;
				}
				if (!$.isFunction(instance[options]) || options.charAt(0) === "_") {
					logError("no such method '" + options + "' for checkin instance");
					return;
				}
				dataToReturn = instance[options].apply(instance, args);
			} else {
				if (instance) {
					instance._init();
				} else {
					instance = $.data(this, 'checkin', new $.CheckIn(options, this));
				}
			}
		});
		return dataToReturn || this;
	};
})(jQuery, window);