<?php

interface iLog {
  public function execute();
}

/**
 * @author Thomas Schfer <thomas.schaefer@query4u.de>
 * Class for creating a simple server-based progress indicator that
 * works in all browsers (yes, it supports google's chrome, too). Useful for many different applications 
 * where you want to show a status of a server-side operation. No ajax needed. Runs in all major browsers 
 * (tested with ff, chrome, safari, opera, internet explorer). 
 * use cases:
 * - file upload
 * - any application running within iframe
 * 
 * @example
 * $buff = new ScriptProgress();
 * do{
 *   if($buff->getIteration() < $buff->getMax()){
 *     $buff->next();
 *   } else {
 *     echo sprintf("<pre>%s</pre>", print_r($_SERVER,true));
 *     break;
 *   }
 *   
 * }while(true);
 */
class ScriptProgress {

  protected $_interval = 150000; // 0.150 sec
  protected $_inc = 0;
  protected $_steps;
  protected $_progess_indicator = ".";
  protected $_browser;
  
  const VERSION = "0.3";
  
  /**
   * @param integer $steps max iterations
   * @param integer $interval interval wait state
   * @param string $_progess_indicator 
   */
  public function __construct($steps = 10, $interval = 150000, $_progess_indicator = ".") {
    $this->_steps = $steps;
    $this->_interval = $interval;
    $this->_progess_indicator = $_progess_indicator;

    ignore_user_abort(TRUE); // run script in background
    set_time_limit(0); // run script forever  
    $this->_quirkMode();

    ob_start();
  }
  
  public function addStylesheet($name="style"){
    echo sprintf('<link href="%s.css" type="text/css" media="screen, projection" rel="stylesheet"/>', $name);
    if(stristr($_SERVER["HTTP_USER_AGENT"], 'MSIE 9')){
      echo '<!--[if gte IE 9]><style type="text/css">div{filter: none;}</style><![endif]-->';
    }
    return $this;
  }
  
  public function addScript($name="scriptprogress.js"){
    echo sprintf('<script type="text/javascript" src="%s"></script>', $name);
    return $this;
  }

  /**
   * javascript notify action
   * @param string $name
   * @param string $result
   * @return ScriptProgress 
   */
  public function notify($name, $result=null){
    if($result){
      echo sprintf('<script type="text/javascript">notify("%s", "%s");</script>', $name, $result);
    } else {
      echo sprintf('<script type="text/javascript">notify("%s");</script>', $name);
    }
    return $this;
  }
  
  /**
   * wrapper for executing log objects
   * @param object $log 
   */
  public function log($log) {
    if(is_object($log) and (method_exists($log, $execute))) {
      $log->execute();
    } else if(is_object($log)) {
      $r = new ReflectionClass(($log));
      if($r->implementsInterface("iLog")) {
        $log->execute();
      }
    }
    
  }
  
  /**
   * inject piece of code to override the indicator string
   * @param type $value
   * @return \ScriptProgress 
   */
  public function set($value){
    $this->_progess_indicator = $value;
    return $this;
  }

  /**
   * write to browser console
   * @param type $string 
   */
  public function console($string) {
    $this->_progess_indicator = sprintf('<script>console.log("%s")</script>', addslashes($string));
    return $this;
  }
  
  /**
   * get max iterations
   * @return int
   */
  public function getMax() {
    return $this->_steps;
  }
  
  public function getIteration() {
    return $this->_inc;
  }
  
  /**
   * increase iterations
   * @param int $inc 
   */
  public function increase($inc = 1){
    $this->_inc += (int) $inc;
    return $this;
  }

  /**
   * decrease iterations
   * @param int $inc 
   */
  public function decrease($inc = 1){
    $this->_inc -= (int) $inc;
    return $this;
  }

  /**
   * next progress indicator 
   */
  public function next() {
    $this->_buffer_flush();
    echo $this->_progess_indicator;
    $this->_buffer_flush();
    usleep($this->_interval);
    $this->_buffer_flush();
    $this->_inc++;
    return $this;
  }

  public function end() {
    $this->set("<p>ScriptProgress v".self::VERSION."</p>");
    $this->next();
    $this->_buffer_flush();
    echo "</div></body></html>";
    $this->_buffer_flush();
    usleep($this->_interval);
    $this->_buffer_flush();
    $this->_inc++;
    
  }
  
  private function _quirkMode() {
    
    $meta = '<meta name="viewport" content="width=device-width, initial-scale=1.25"/>';

    if (
            strpos($_SERVER["HTTP_USER_AGENT"], "Gecko") or
            strpos($_SERVER["HTTP_USER_AGENT"], "WebKit")
    ) {
            
      $this->_browser = 'gecko';
      echo '<?xml version="1.0" encoding="iso-8859-1"?><!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">';
      echo '<html><head><title>Script Progress v'.self::VERSION.'</title>'.$meta.'</head><body><div class="center" id="center">';
    }
    elseif (
            strpos($_SERVER["HTTP_USER_AGENT"], "MSIE")
    ) {
      
      $this->_browser = 'ie';
      echo '<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.0 Strict//EN"><html><head><title>Script Progress</title></head><body>';
      echo '<html><head><title>Script Progress v'.self::VERSION.'</title>'.$meta.'</head><body><div class="center" id="center">';
    }
    elseif (
            stristr($_SERVER["HTTP_USER_AGENT"], "Opera")
    ) {
      $this->_browser = 'opera';
      echo '<?xml version="1.0" encoding="utf-8"?><!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">';
      # opera hack
      echo '<html><head><title>Script Progess v'.self::VERSION.'</title>'.$meta.'</head><body><div class="center" id="center">';
    }
  }

  /**
   * write some browser-related strings to fill up content header
   */
  private function _buffer_flush() {
    switch($this->_browser)
    {
      default:
        echo "<!-- Deal with browser-related buffering by sending some incompressible strings -->";
        for ($i = 0; $i < 5; $i++) {
          echo "<!-- abcdefghijklmnopqrstuvwxyz1234567890aabbccddeeffgghhiijjkkllmmnnooppqqrrssttuuvvwwxxyyzz11223344556677889900abacbcbdcdcededfefegfgfhghgihihjijikjkjlklkmlmlnmnmononpopoqpqprqrqsrsrtstsubcbcdcdedefefgfabcadefbghicjkldmnoepqrfstugvwxhyz1i234j567k890laabmbccnddeoeffpgghqhiirjjksklltmmnunoovppqwqrrxsstytuuzvvw0wxx1yyz2z113223434455666777889890091abc2def3ghi4jkl5mno6pqr7stu8vwx9yz11aab2bcc3dd4ee5ff6gg7hh8ii9j0jk1kl2lmm3nnoo4p5pq6qrr7ss8tt9uuvv0wwx1x2yyzz13aba4cbcb5dcdc6dedfef8egf9gfh0ghg1ihi2hji3jik4jkj5lkl6kml7mln8mnm9ono -->";
        }

        while (ob_get_level()) {
          ob_end_flush();
        }

        if (ob_get_length()) {
          @ob_flush();
          @flush();
          @ob_end_flush();
        }
        @ob_start();
        break;
    }
  }

}
