﻿using System;
using System.Web;
using System.IO;
using System.Web.Mvc;
//
using MvcBasic.Logic;

namespace MvcBasicSite.Models.Helpers
{
    /// <summary>
    /// Defines the image rezult action used for rendering an image from a given stream.
    /// </summary>
    public class ImageResult : ActionResult
    {
        /// <summary>
        /// Gets or sets the image stream.
        /// </summary>
        public Stream ImageStream { get; private set; }
        /// <summary>
        /// Gets or sets the content type.
        /// </summary>
        public string ContentType { get; private set; }

        /// <summary>
        /// Initializes a new instance of the ImageResult class.
        /// </summary>
        /// <param name="imageStream">The image stream.</param>
        /// <param name="contentType">The content type (mandatory).</param>
        public ImageResult(Stream imageStream, string contentType)
        {
            if (contentType == null)
                throw new ArgumentNullException("contentType");
            //
            this.ImageStream = imageStream;
            this.ContentType = contentType;
        }

        /// <summary>
        /// Read the image from the stream then render it.
        /// </summary>
        /// <param name="context">The context (controller, HTTP content, request context, and route data) in which the result is executed.</param>
        public override void ExecuteResult(ControllerContext context)
        {
            if (context == null)
                throw new ArgumentNullException("context");
            //
            try
            {
                HttpResponseBase response = context.HttpContext.Response;
                response.ContentType = this.ContentType;
                //
                if (this.ImageStream == null)
                {
                    string filePath = context.HttpContext.Server.MapPath("/Content/Images/noimageSmall.jpg");
                    System.Drawing.Image imageIn = System.Drawing.Image.FromFile(filePath);
                    MemoryStream ms = new MemoryStream();
                    //
                    imageIn.Save(ms, System.Drawing.Imaging.ImageFormat.Jpeg);
                    response.OutputStream.Write(ms.ToArray(), 0, (int)ms.Length);
                }
                else
                {
                    byte[] buffer = new byte[4096];
                    //
                    while (true)
                    {
                        int read = this.ImageStream.Read(buffer, 0, buffer.Length);
                        if (read == 0)
                            break;
                        //
                        response.OutputStream.Write(buffer, 0, read);
                    }
                }
                //
                response.End();
            }
            catch (Exception ex)
            {
                MvcBasicLog.LogException(ex);
            }
        }
    }

    /// <summary>
    /// Defines the controller extensions used to generate the image results.
    /// </summary>
    public static class ControllerExtensions
    {
        public static ImageResult Image(this Controller controller, Stream imageStream, string contentType)
        {
            return new ImageResult(imageStream, contentType);
        }

        public static ImageResult Image(this Controller controller, byte[] imageBytes, string contentType)
        {
            if(imageBytes == null || imageBytes.Length == 0)
                return new ImageResult( null , contentType);
            else
                return new ImageResult(new MemoryStream(imageBytes), contentType);
        }
    }
}