﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Data.Objects;

namespace MvcBasic.Logic
{
    /// <summary>
    /// Defines the visitors log entity.
    /// </summary>
    public partial class VisitorLog
    {
        /// <summary>
        /// Create a new visitor log entry when user start using the web shop.
        /// </summary>
        /// <param name="dataContext">The data context.</param>
        /// <param name="userID">The user ID or 0.</param>
        /// <returns>The visitor log ID. </returns>
        public static int OnUserLogin(MvcBasicSiteEntities dataContext, int userID)
        {
            VisitorLog newEntry = new VisitorLog();
            newEntry.StartDate = DateTime.Now;
            newEntry.EndDate = newEntry.StartDate; // To check on UI!
            newEntry.UserID = (userID > 0 ? (int?)userID : null);
            //
            dataContext.VisitorLogs.AddObject(newEntry);
            dataContext.SaveChanges();
            //
            return newEntry.ID;
        }

        /// <summary>
        /// Update the visitor log entry when user logoff or timeout.
        /// </summary>
        /// <param name="dataContext">The data context.</param>
        /// <param name="visitorLogID">The visitor log ID.</param>
        /// <param name="wasTimeout">Was timeout flag.</param>
        /// <returns>True for success.</returns>
        public static bool OnUserLogoff(MvcBasicSiteEntities dataContext, int visitorLogID, bool wasTimeout)
        {
            if (visitorLogID == 0)
            {
                //
                // Anonymous Log-Off.
                //
                VisitorLog newEntry = new VisitorLog();
                newEntry.StartDate = DateTime.Now;
                newEntry.EndDate = newEntry.StartDate;
                newEntry.UserID = null;
                if (wasTimeout)
                    newEntry.WasTimeOut = wasTimeout;
                //
                dataContext.VisitorLogs.AddObject(newEntry);
                dataContext.SaveChanges();
                //
                return true;
            }
            else
            {
                // 
                // Normal User Log-Off.
                //
                VisitorLog visitorLog = dataContext.VisitorLogs.FirstOrDefault(item => item.ID == visitorLogID);
                if (visitorLog == null)
                    return false;
                //
                visitorLog.EndDate = DateTime.Now;
                if (wasTimeout)
                    visitorLog.WasTimeOut = wasTimeout;
                //
                dataContext.SaveChanges();
                //
                return true;
            }
        }

        /// <summary>
        /// Quick search visitor logs .
        /// </summary>
        /// <param name="dataContext">The data context.</param>
        /// <param name="searchStartDate">The search start date.</param>
        /// <param name="searchEndDate">The search end date.</param>
        /// <returns></returns>
        public static IQueryable<VisitorLog> SearchLogByDates(MvcBasicSiteEntities dataContext, DateTime searchStartDate, DateTime searchEndDate)
        {
            if (searchStartDate.Date == DateTime.MinValue && searchEndDate.Date == DateTime.MinValue)
            {
                return dataContext.VisitorLogs.AsQueryable();
            }
            else if (searchStartDate.Date == DateTime.MinValue)
            {
                var searchResults = dataContext.VisitorLogs.Where(c => 
                    EntityFunctions.TruncateTime(c.StartDate) <= EntityFunctions.TruncateTime(searchEndDate));
                //
                return searchResults.AsQueryable();
            }
            else if (searchEndDate.Date == DateTime.MinValue)
            {
                var searchResults = dataContext.VisitorLogs.Where(c => 
                    EntityFunctions.TruncateTime(c.StartDate) >= EntityFunctions.TruncateTime(searchStartDate));
                //
                return searchResults.AsQueryable();
            }
            else
            {
                var searchResults = dataContext.VisitorLogs.Where(c =>
                    EntityFunctions.TruncateTime(c.StartDate) >= EntityFunctions.TruncateTime(searchStartDate) &&
                    EntityFunctions.TruncateTime(c.StartDate) <= EntityFunctions.TruncateTime(searchEndDate));
                //
                return searchResults.AsQueryable();
            }
        }
        /// <summary>
        /// Delete the selected visitor logs.
        /// </summary>
        /// <param name="dataContext">The data context.</param>
        /// <param name="searchStartDate">The search start date</param>
        /// <param name="searchEndDate">The search end date.</param>
        /// <returns></returns>
        public static void DeleteLogByDates(MvcBasicSiteEntities dataContext, DateTime searchStartDate, DateTime searchEndDate)
        {
            if (searchStartDate.Date == DateTime.MinValue && searchEndDate.Date == DateTime.MinValue)
            {
                foreach (VisitorLog visitorLog in dataContext.VisitorLogs)
                {
                    dataContext.DeleteObject(visitorLog);
                }
            }
            else if (searchStartDate.Date == DateTime.MinValue)
            {
                var searchResults = dataContext.VisitorLogs.Where(c =>
                    EntityFunctions.TruncateTime(c.StartDate) <= EntityFunctions.TruncateTime(searchEndDate));
                //
                DeleteSelectedLogs(dataContext, searchResults);

            }
            else if (searchEndDate.Date == DateTime.MinValue)
            {
                var searchResults = dataContext.VisitorLogs.Where(c => 
                    EntityFunctions.TruncateTime(c.StartDate) >= EntityFunctions.TruncateTime(searchStartDate));
                //
                DeleteSelectedLogs(dataContext, searchResults);
            }
            else
            {
                var searchResults = dataContext.VisitorLogs.Where(c =>
                    EntityFunctions.TruncateTime(c.StartDate) >= EntityFunctions.TruncateTime(searchStartDate) &&
                    EntityFunctions.TruncateTime(c.StartDate) <= EntityFunctions.TruncateTime(searchEndDate));
                //
                DeleteSelectedLogs(dataContext, searchResults);
            }
            //
            dataContext.SaveChanges();
        }

        /// <summary>
        /// Delete the selected logs.
        /// </summary>
        /// <param name="dataContext">The data context.</param>
        /// <param name="searchResults">The search results.</param>
        private static void DeleteSelectedLogs(MvcBasicSiteEntities dataContext, IEnumerable<VisitorLog> searchResults)
        {
            foreach (var item in searchResults)
            {
                dataContext.VisitorLogs.DeleteObject(item);
            }
        }
    }
}
