/* description: Parses end executes mathematical expressions. */

/* lexical grammar */
%lex
%%

\s+                   /* skip whitespace */
[0-9]+("."[0-9]+)?\b  return 'NUMBER'
"*"                   return '*'
"/"                   return '/'
"MOD"                 return 'MOD'
"-"                   return '-'
"+"                   return '+'
"^"                   return '^'
"SQRT"                return 'SQRT'
"("                   return '('
")"                   return ')'
","                   return ','
">"                   return '>'
"<"                   return '<'
"="                   return '='
"PI"                  return 'PI'
"E"                   return 'E'
"IF"                  return 'IF'
"ABS"                 return 'ABS'
"ROUND"               return 'ROUND'
"FLOOR"               return 'FLOOR'
"CEIL"                return 'CEIL'
"MAX"		      return 'MAX'
"MIN"		      return 'MIN'
"AVG"		      return 'AVG'
"SUM"		      return 'SUM'
<<EOF>>               return 'EOF'
.                     return 'INVALID'

/lex

/* operator associations and precedence */

%left ',' '='
%left '<=' '>=' '<>' 
%left '>' '<'
%left '+' '-' 'SUM' 'AVG' 'MIN' 'MAX'
%left '*' '/' 'MOD'
%left '^'
%left 'ABS' 'ROUND' 'CEIL' 'FLOOR' 'IF' 
%left UMINUS

%start expressions

%% /* language grammar */

expressions
    : e EOF
        {return $1;}
    ;

e
    : e '+' e
        {$$ = $1+$3;}
    | e '-' e
        {$$ = $1-$3;}
    | e '*' e
        {$$ = $1*$3;}
    | e '/' e
        {$$ = $1/$3;}
    | e '>' e
        {$$ = $1 > $3}
    | e '<' e
        {$$ = $1 < $3}
    | e '>''=' e
        {$$ = $1 >= $4}
    | e '<''=' e
        {$$ = $1 <= $4}
    | e '=' e
        {$$ = $1 == $3}
    | e '<''>' e
        {$$ = $1 != $4}
    | e '^' e
        {$$ = Math.pow($1, $3);}
    | e 'MOD' e
        {$$ = $1%$3;}
    | '(' e ')'
        {$$ = $2;}
    | 'ABS''(' e ')'
        {$$ = Math.abs($3);}
    | 'ROUND''(' e ')'
        {$$ = Math.round($3);}
    | 'FLOOR''(' e ')'
        {$$ = Math.floor($3);}
    | 'CEIL''(' e ')'
        {$$ = Math.ceil($3);}
    | 'SQRT''(' e ')'
        {$$ = Math.sqrt($3);}
    | 'IF' '(' e ',' e ',' e ')'
        {$$ = ($3) ? $5 : $7;}
    | 'MAX' '(' e ',' e ')'
        {$$ = formula.max($3,$5);}
    | 'MIN' '(' e ',' e ')'
        {$$ = formula.min($3,$5);}
    | 'AVG' '(' e ',' e ')'
        {$$ = formula.avg($3,$5);}
    | 'SUM' '(' e ',' e ')'
        {$$ = formula.sum($3,$5);}
    | '-' e %prec UMINUS
        {$$ = -$2;}
    | NUMBER
        {$$ = Number(yytext);}
    | E
        {$$ = Math.E;}
    | PI
        {$$ = Math.PI;}
    ;
