<?php
/**
 * OnHTML generator class. 
 * @author Matiss Roberts Treinis <roberts@x0.lv>
 * @copyright 2012 Matiss Roberts Treinis <roberts@x0.lv>
 * @version 
 * 
 */
class OnHTML{

	/**
	 * This variable contains all the HTML data of this element. Either a string or array of child elements.
	 *
	 * @var string 
	 *
	 */	
	public $html;
	
	
	/**
	 * Default tag used when creating a new element
	 *
	 * @var mixed 
	 *
	 */	
	public $tag = 'div';
	
	
	/**
	 * Unique identificator of current element
	 *
	 * @var string 
	 *
	 */	
	private $__oid__;
	
	
	/**
	 * Should we enable indexing. Default is false - disabled. Large element sets will degrade performance, and index should be disabled when generating bulk trees.
	 *
	 * @var bool 
	 *
	 */	
	private $must_index = false;
	
	/* 
	*	Do not uncommet the following property's. 
	*	They are preserved only for reference, uncommenting them
	*	will lead to slightly increased size of the JSON output for
	*	elements where those property's are uninitialized.
	*/
	///public $attributes  = null;
	///public $callback = null;
	///public $events = null;
	///public $css = null;
	
	
	/**
	 * Construct a new HTML element.
	 *
	 * @param string $html Contents of this element - HTML text or onHTML object.
	 * @param string $tag Tag of the current element.
	 * @return OnHTML new OnHTML element.
	 *
	 */	
	function __construct($html = null,$tag = 'div'){
		global $_ONHTML;
		if($_ONHTML == null){
			$_ONHTML = array();
		}
		$this->html = $html;
		$this->tag = $tag;
		$this->__oid__ = uniqid();
		if($this->must_index){
			$this->appendIndex('__onhtml_i_global',$this);
		}
		return $this;
	}
	
	/**
	 * Set the indexing on or off for current object.
	 * Indexing will be inherited within elements appended to this element.
	 * with addElement.
	 *
	 * @param bool $state True = on, false = off.
	 * @return OnHTML Current object.
	 *
	 */	
	function setIndexing($state){
		if(is_bool($state)){
			$this->must_index = $state;
		}
		return $this;
	}
	
	/**
	 * Add a new child element to current element.
	 *
	 * @param mixed $html Contents of the new element - text or OnHTML object.
	 * @param string $tag Tag of the new element.
	 * @param string $id ID of the new element. Will be assigned as HTML attribute "id". Childs of the same element with same id will replace old element with old id.
	 * @return OnHTML Reference of the element created.
	 *
	 */	
	function addElement($html = null,$tag = 'div',$id = null){
		global $_ONHTML;
		$element = new OnHTML($html,$tag);
		$element->must_index = $this->must_index;
		
		if($this->must_index){
			$this->appendIndex('__onhtml_i_tag',$element,$tag);
		}
		
		if(is_array($this->html) == false){
			$this->html = array();
		}
		
		if($id === null){
			$this->html[] = $element;
		}else{
			$this->html[$id] = $element;
			$element->attr('id',$id);
			if($this->must_index){
				$this->appendIndex('id',$element,$id);
			}
		}
		return $element;
	}
	
	/**
	 * Add a child element to current element.
	 *
	 * @param OnHTML $element Element to add.
	 * @param mixed $id ID of the element. Element with ID already assigned to child of the current element will replace the old element with same id. If id will not be set, ID attribute of the appendable element will be used.
	 * @return OnHTML Reference of the current object.
	 *
	 */	
	function addChild(OnHTML $element,$id = null){
		global $_ONHTML;
		if($this->must_index){
			$this->appendIndex('__onhtml_i_tag',$element,$element->tag);
		}
		if(is_array($this->html) == false){
			$this->html = array();
		}
		
		if($id === null && isset($element->attributes['id'])){
			$id = $element->attributes['id'];
		}
		
		if($id === null){
			$this->html[] = $element;
		}else{
			$this->html[$id] = $element;
			$this->attr('id',$id);
			if($this->must_index){
				$this->appendIndex('id',$element,$id);
			}
		}
		
		return $this;
	}
	
	/**
	 * Register a frontend Javascript function as the load callback of the current element.
	 *
	 * @param string $name Name of the function.
	 * @param array $args Optional arguments to pass to the function.
	 * @return OnHTML Reference of the current object.
	 *
	 */	
	function addLoadCallback($name,array $args = array()){
		$this->callback[$name] = $args;
		return $this;
	}
	
	/**
	 * Add one or multiple HTML classes to current element. You can pass infinite amount of arguments or one argument - array, containing all the classes. 
	 *
	 * @return OnHTML Reference of the current object.
	 *
	 */	
	function addClass($classes){
		if(!is_array($classes)){
			$classes = func_get_args();
		}
		
		if(!isset($this->attr['class'])){
			$this->attr['class'] = implode(' ',$classes);
			if($this->must_index){
				foreach($classes as $class){
					$this->appendIndex('class',$this,$class);
				}
			}
		}else{
			$old_classes = explode(' ',$this->attr['class']);
			$new_classes = array_merge($old_classes,$classes);
			$this->attr['class'] = implode(' ',$new_classes);
			if($this->must_index){
				foreach($new_classes as $class){
					$this->appendIndex('class',$this,$class);
				}
			}
		}
		return $this;
	}
	
	/**
	 * Remove one or more classes from current element. You can pass infinite amount of arguments to this function or one array containing all the classes to remove.
	 *
	 * @return OnHTML Reference of the current object.
	 *
	 */	
	function removeClass($candidates){
		if(is_array($candidates) == false){
			$candidates = func_get_args();
		}		
		if(count($candidates) !== 0){
			$classes = explode(' ',$this->attr['class']);
			$diff = array_diff($classes,$candidates);
			$this->attr['class'] = implode(' ',$diff);
			if($this->must_index){
				foreach($candidates as $candidate){
					$this->removeFromIndex('class',$this,$candidate);
				}
			}
		}
		return $this;
	}
	
	/**
	 * Add one or array of attributes to the current element. 
	 *
	 * @param mixed $key Name of the attribute or array of attributes, where key is attribute name and value is attribute value.
	 * @param string $value Value of the attribute.
	 * @return OnHTML Reference of the current object.
	 *
	 */	
	function attr($key,$value = null){
		if(is_array($key)){
			foreach($key as $akey=>$value){
				$this->attributes[$akey] = $value;
				if($this->must_index){
					$this->appendIndex($akey,$this,$value);
				}
			}
		}else{
			$this->attributes[$key] = $value;
			if($this->must_index){
				$this->appendIndex($key,$this,$value);
			}
		}
		return $this;
	}
	
	/**
	 * Remove one or more attributes from current object. You can pass infinite amount of arguments - attribute names to be removed or one argument, array that contains all the names to remove.
	 *
	 * @return OnHTML Reference of the current object.
	 *
	 */	
	function removeAttr($candidates){
		if(is_array($candidates) == false){
			$candidates = func_get_args();
		}
		foreach($candidates as $name){
			if(isset($this->attributes[$name])){
				unset($this->attributes[$name]);
				if($this->must_index){
					$this->removeFromIndex($name,$this,$name);
				}
			}
		}
		return $this;
	}
	
	/**
	 * Bind one or more event handlers for current element. You can pass array as first element where key is event name and value is event function handler name. Event name must be jQuery compatible.
	 *
	 * @param mixed $event Name of the event or array containing event name as item key and function names as item values.
	 * @param string $function Function to be assigned as event callback.
	 * @return OnHTML Reference of the current object.
	 *
	 */	
	function bind($event,$function = null){
		if(is_array($event)){
			foreach($event as $key=>$func){
				$this->events[$key] = $func;
			}
		}else{
			$this->events[$event] = $function;
		}
		return $this;
	}
	
	
	/**
	 * Assign one or more element CSS property's.
	 *
	 * @param mixed $prop Name of the css property or array containing multiple property names as item keys and values as item values.
	 * @param string $value Value of the css property.
	 * @return OnHTML Reference of the current object.
	 *
	 */	
	function css($prop,$value = null){
		if(is_array($prop)){
			foreach($prop as $key=>$value){
				$this->css[$key] = $value;
			}
		}else{
			$this->css[$prop] = $value;
		}
		return $this;
	}
	
	/**
	 * Get array of elements by class name. Indexing must be enabled for this method to work.
	 *
	 * @param string $key Class name.
	 * @return mixed Array of elements found, false if no element match the class name.
	 *
	 */	
	function getElementsByClassName($key){
		global $_ONHTML;
		if(isset($_ONHTML['index']['class'][$key]) && count($_ONHTML['index']['class'][$key]) != 0){
			return $_ONHTML['index']['class'][$key];
		}else{
			return false;
		}
	}
	
	/**
	 * Get array of elements by tag name. Indexing must be enabled for this method to work.
	 *
	 * @param string $key Tag name.
	 * @return mixed Array of elements found, false if no element match the tag name.
	 *
	 */	
	function getElementsByTagName($key){
		global $_ONHTML;
		if(isset($_ONHTML['index']['tag'][$key])){
			return $_ONHTML['index']['tag'][$key];
		}else{
			return false;
		}
	}
	
	/**
	 * Get single element by ID of the element. Indexing must be enabled for this method to work.
	 *
	 * @param string $key ID of the element. 
	 * @return mixed OnHTML element or false if no element match the ID.
	 *
	 */	
	function getElementById($key){
		global $_ONHTML;
		if(isset($_ONHTML['index']['id'][$key])){
			return $_ONHTML['index']['id'][$key];
		}else{
			return false;
		}
	}
	
	/**
	 * Get single element by Object ID of the element. Indexing must be enabled for this method to work.
	 *
	 * @param string $key OID to search for.
	 * @return mixed OnHTML element or false if no element match the ID.
	 *
	 */	
	function getElementByOID($key){
		global $_ONHTML;
		if(isset($_ONHTML['index']['global'][$key])){
			return $_ONHTML['index']['global'][$key];
		}else{
			return false;
		}
	}	
	
	/**
	 * Register element to global element index.
	 *
	 * @param string $index Index type.
	 * @param OnHTML $element Element to register.
	 * @param string $key Key of the element to index.
	 * @return void
	 *
	 */	
	private function appendIndex($index,$element,$key = null){
		global $_ONHTML;
		if($index == 'id'){
			$_ONHTML['index']['id'][$key] = $element->tag;	
		}elseif($index == 'class'){
			$_ONHTML['index']['class'][$key][$element->__oid__] = $element->tag;
		}elseif($index == '__onhtml_i_tag'){
			$_ONHTML['index']['tag'][$key][$element->__oid__] = $element->tag;
		}elseif($index == '__onhtml_i_global'){
			$_ONHTML['index']['global'][$element->__oid__] = $element->tag;
		}
	}
	
	/**
	 * Remove element from the index.
	 *
	 * @param string $index Index type.
	 * @param OnHTML $element Element to remove.
	 * @param string $key Key of the element to remove.
	 * @return void 
	 *
	 */	
	private function removeFromIndex($index,$element,$key){
		global $_ONHTML;
		if($index == 'id'){
			if(isset($_ONHTML['index']['id'][$key])){
				unset($_ONHTML['index']['id'][$key]);
			}
		}elseif($index == 'class'){
			if(isset($_ONHTML['index']['class'][$key][$element->__oid__])){
				unset($_ONHTML['index']['class'][$key][$element->__oid__]);
			}
			if(count($_ONHTML['index']['class'][$key]) == 0){
				unset($_ONHTML['index']['class'][$key]);
			}
		}
	}
	
	
	/**
	 * Return the JSON representation of current element and all of the child elements.
	 *
	 * @return string JSON data.
	 *
	 */	
	function json(){
		return json_encode($this);
	}
	
	
}

