﻿using ProductsLibrary.Model;
using System;
using System.Collections.Generic;
using System.IO;
using System.Net.Http.Formatting;
using System.Net.Http.Headers;
using System.ServiceModel.Syndication;
using System.Threading.Tasks;
using System.Xml;

namespace WebApi_CustomMediaTypeFormatter
{
    public class CustomSyndicationFeedFormatter : MediaTypeFormatter
    {
        private readonly string atom = "application/atom+xml";
        private readonly string rss = "application/rss+xml";

        public CustomSyndicationFeedFormatter()
        {
            SupportedMediaTypes.Add(new MediaTypeHeaderValue(atom));
            SupportedMediaTypes.Add(new MediaTypeHeaderValue(rss));
        }

        public override bool CanWriteType(Type type)
        {
            if (type == typeof(FeedItem) || type == typeof(IEnumerable<FeedItem>))
                return true;
            else
                return false;
        }

        public override bool CanReadType(Type type)
        {
            return false;
        }

        public override Task WriteToStreamAsync(Type type, object value, Stream stream, System.Net.Http.HttpContent content, System.Net.TransportContext transportContext)
        {
            return Task.Factory.StartNew(() =>
            {
                if (type == typeof(FeedItem) || type == typeof(IEnumerable<FeedItem>))
                    BuildSyndicationFeed(value, stream, content.Headers.ContentType.MediaType);
            });
        }

        private void BuildSyndicationFeed(object models, Stream stream, string contenttype)
        {
            List<SyndicationItem> items = new List<SyndicationItem>();
            var feed = new SyndicationFeed()
            {
                Title = new TextSyndicationContent("Custom Feeds")
            };

            if (models is IEnumerable<FeedItem>)
            {
                var enumerator = ((IEnumerable<FeedItem>)models).GetEnumerator();
                while (enumerator.MoveNext())
                {
                    items.Add(BuildSyndicationItem(enumerator.Current));
                }
            }
            else
            {
                items.Add(BuildSyndicationItem((FeedItem)models));
            }

            feed.Items = items;

            using (XmlWriter writer = XmlWriter.Create(stream))
            {
                if (string.Equals(contenttype, atom))
                {
                    Atom10FeedFormatter atomformatter = new Atom10FeedFormatter(feed);
                    atomformatter.WriteTo(writer);
                }
                else
                {
                    Rss20FeedFormatter rssformatter = new Rss20FeedFormatter(feed);
                    rssformatter.WriteTo(writer);
                }
            }
        }

        private SyndicationItem BuildSyndicationItem(FeedItem feedItem)
        {
            var category = new SyndicationCategory { Label = feedItem.Category };
          
            var item = new SyndicationItem()
            {
                Title = new TextSyndicationContent(feedItem.Title),               
                LastUpdatedTime = feedItem.CreatedAt,
                Summary = new TextSyndicationContent(feedItem.Description) ,
            };

            item.Categories.Add(category);
            item.Authors.Add(new SyndicationPerson() { Name = feedItem.CreatedBy });

            return item;
        }     
    }
}
