﻿#region Copyright (c) 2010 Raul Iloc
/*
{***************************************************************************}
{                                                                           }
{       Copyright (c) 2010  RAUL ILOC (rauliloc@yahoo.com)                  }
{       ALL RIGHTS RESERVED                                                 }
{                                                                           }
{   THE WORK IS PROVIDED UNDER THE TERMS OF THIS CODE PROJECT OPEN LICENSE. }
{   THE WORK IS PROTECTED BY COPYRIGHT AND/OR OTHER APPLICABLE LAW.         }
{   ANY USE OF THE WORK OTHER THAN AS AUTHORIZED UNDER THIS LICENSE         }
{   OR COPYRIGHT LAW IS PROHIBITED.                                         }
{                                                                           }
{                                                                           }
{  BY EXERCISING ANY RIGHTS TO THE WORK PROVIDED HEREIN, YOU ACCEPT AND     }
{  AGREE TO BE BOUND BY THE TERMS OF THIS LICENSE. THE AUTHOR GRANTS YOU    }
{  THE RIGHTS CONTAINED HEREIN IN CONSIDERATION OF YOUR ACCEPTANCE OF SUCH  }
{  TERMS AND CONDITIONS. IF YOU DO NOT AGREE TO ACCEPT AND BE BOUND BY THE  }
{  TERMS OF THIS LICENSE, YOU CANNOT MAKE ANY USE OF THE WORK.              }
{                                                                           }
{***************************************************************************}
*/
#endregion Copyright (c) 2010 Raul Iloc

using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Data.Objects;
//
using Ra.GridView.Data;
using Ra.GridView.Util;

namespace Ra.GridView.Web.Data
{
    /// <summary>
    /// Defines the contact list page data.
    /// </summary>
    /// <remarks>
    /// Used to implement optimized pagination and sorting in ContactListPage page. 
    /// </remarks>
    public class ContactListPageData
    {
        /// <summary>
        /// The associated page.
        /// </summary>
        private static ContactListPage _page;
        /// <summary>
        /// Gets or sets the associated page.
        /// </summary>
        public static ContactListPage Page
        {
            get { return _page; }
            set { _page = value; }
        }
        /// <summary>
        /// The count af all reacords from the database that match the current searching filter.
        /// </summary>
        private static int _count;
        /// <summary>
        /// The contact ID.
        /// </summary>
        private static int _contactID = 0;
        /// <summary>
        /// Gets or sets the contact ID.
        /// </summary>
        public static int ContactID
        {
            get { return _contactID; }
            set { _contactID = value; }
        }
        /// <summary>
        /// The new filter flag.
        /// </summary>
        private static bool _newFilter = false;
        /// <summary>
        /// Gets or sets the new filter flag.
        /// </summary>
        public static bool NewFilter
        {
            get { return _newFilter; }
            set { _newFilter = value; }
        }
        /// <summary>
        /// The after delete flag.
        /// </summary>
        private static bool _afterDelete = false;
        /// <summary>
        /// Gets or sets the after delete flag.
        /// </summary>
        public static bool AfterDelete
        {
            get { return _afterDelete; }
            set { _afterDelete = value; }
        }

        /// <summary>
        /// Get the count af all reacords from the database that match the current searching filter.
        /// </summary>
        /// <returns>The count af all reacords from the database that match the current searching filter.</returns>
        public static int GetCount()
        {
            return (_afterDelete ? _count - 1 : _count);
        }

        /// <summary>
        /// Gets the data by using the current filter and soting from the associated page.
        /// </summary>
        /// <param name="startIndex">The data start index (into all results).</param>
        /// <param name="pageSize">The page size.</param>
        /// <param name="sortBy">The sorting expresion.</param>
        /// <returns>The data results.</returns>
        public static DataTable GetDataByFilter(int startIndex, int pageSize, string sortBy)
        {
            try
            {
                RaGridViewEntities dataContext = _page.DataContext;
                List<Contact> contactList = new List<Contact>();
                //
                if (_contactID > 0)
                {
                    //
                    // Load only one entity by ID!
                    //
                    Contact contact = dataContext.GetContactByID(_contactID).SingleOrDefault();
                    if (contact == null || contact.Deleted == true)
                    {
                        //
                        //  No results!
                        //
                        _count = 0;
                    }
                    else
                    {
                        contactList = new List<Contact>() { contact };
                        _count = 1;
                    }
                }
                else if (_contactID < 0)
                {
                    //
                    //  No results!
                    //
                    _count = 0;
                }
                else
                {
                    //
                    // Prepare the sorting expression
                    //
                    if (string.IsNullOrEmpty(sortBy))
                        sortBy = "c.ID DESC";
                    else if (sortBy.EndsWith(" DESC"))
                    {
                        sortBy = sortBy.Replace(",", " DESC,");
                    }
                    //
                    // Search data into database for the current filter and current page
                    //
                    int oldCount = _count;
                    int realIndex = (_newFilter ? 0 : startIndex);
                    //
                    if (_afterDelete && realIndex == _count - 1)
                        realIndex = (realIndex > pageSize ? realIndex - pageSize + 1 : 1);
                    else
                        realIndex++;
                    //
                    ObjectParameter countParameter = new ObjectParameter("count", typeof(int));
                    //
                    contactList = dataContext.GetContactsByFilterPaginated(realIndex, pageSize, sortBy,
                        _page.GroupID, countParameter).ToList<Contact>();
                    _count = (int)countParameter.Value;
                    //
                    if (oldCount > _count)
                    {
                        //
                        // New records were added into the database (for example by other users) 
                        // so you have to reset the current page index and reload data!
                        //
                        realIndex = 1;
                        contactList = dataContext.GetContactsByFilterPaginated(realIndex, pageSize, sortBy,
                            _page.GroupID, countParameter).ToList<Contact>();
                        _count = (int)countParameter.Value;
                    }
                }
                //
                return ListToDataTable(contactList);
            }
            catch (Exception ex)
            {
                RaGridViewEventLog.LogException(ex);
                _page.ErrorMessage = "Error in loading the data from the database!";
            }
            finally
            {
                //
                // Clear the used flags!
                //
                _afterDelete = false;
                _newFilter = false;
            }
            //
            // Convert the results into a datatable
            //
            return ListToDataTable(new List<Contact>());
        }

        /// <summary>
        /// Converts a list of entities into a datatable used for gridview data binding.
        /// </summary>
        /// <param name="entityList">The entities list.</param>
        /// <returns>The data results as datatable.</returns>
        public static DataTable ListToDataTable(List<Contact> entityList)
        {
            DataTable contactsDataTable = new DataTable();
            //
            contactsDataTable.Columns.Add("ID", typeof(int));
            contactsDataTable.Columns.Add("Person", typeof(string));
            contactsDataTable.Columns.Add("Group", typeof(string));
            contactsDataTable.Columns.Add("Phone", typeof(string));
            contactsDataTable.Columns.Add("Email", typeof(string));
            //
            foreach (Contact contact in entityList)
            {
                contactsDataTable.Rows.Add(contact.ID,
                    String.Format("{0} {1}", contact.FirstName, contact.LastName),
                    (contact.Group == null ? string.Empty : contact.Group.Name),
                    (contact.Phone == null ? string.Empty : contact.Phone),
                    (contact.Email == null ? string.Empty : contact.Email));
            }
            //
            return contactsDataTable;
        }
    }
}