﻿#region Copyright (c) 2010 Raul Iloc
/*
{***************************************************************************}
{                                                                           }
{       Copyright (c) 2010  RAUL ILOC (rauliloc@yahoo.com)                  }
{       ALL RIGHTS RESERVED                                                 }
{                                                                           }
{   THE WORK IS PROVIDED UNDER THE TERMS OF THIS CODE PROJECT OPEN LICENSE. }
{   THE WORK IS PROTECTED BY COPYRIGHT AND/OR OTHER APPLICABLE LAW.         }
{   ANY USE OF THE WORK OTHER THAN AS AUTHORIZED UNDER THIS LICENSE         }
{   OR COPYRIGHT LAW IS PROHIBITED.                                         }
{                                                                           }
{                                                                           }
{  BY EXERCISING ANY RIGHTS TO THE WORK PROVIDED HEREIN, YOU ACCEPT AND     }
{  AGREE TO BE BOUND BY THE TERMS OF THIS LICENSE. THE AUTHOR GRANTS YOU    }
{  THE RIGHTS CONTAINED HEREIN IN CONSIDERATION OF YOUR ACCEPTANCE OF SUCH  }
{  TERMS AND CONDITIONS. IF YOU DO NOT AGREE TO ACCEPT AND BE BOUND BY THE  }
{  TERMS OF THIS LICENSE, YOU CANNOT MAKE ANY USE OF THE WORK.              }
{                                                                           }
{***************************************************************************}
*/
#endregion Copyright (c) 2010 Raul Iloc

using System;
using System.Diagnostics;

namespace Ra.GridView.Util
{
    /// <summary>
    /// Defines the class used for logging into the event log. 
    /// </summary>
    public class RaGridViewEventLog
    {
        /// <summary>
        /// The even log source name used by RaGridView system.
        /// </summary>
        private static readonly string _logSource = "Ra.GridView";

        /// <summary>
        /// Write a message into the event log.
        /// </summary>
        /// <param name="message">The message.</param>
        public static void LogMessage(string message)
        {
            LogMessage(null, message);
        }

        /// <summary>
        /// Write a message into the event log.
        /// </summary>
        /// <param name="category">The category (like a project name).</param>
        /// <param name="message">The message.</param>
        public static void LogMessage(string category, string message)
        {
            if (category == null)
                AddLogLine(message, false);
            else
                AddLogLine(string.Format("[{0}] : {1}", category, message), false);
        }

        /// <summary>
        /// Write an error message into the event log.
        /// </summary>
        /// <param name="message">The message.</param>
        public static void LogError(string message)
        {
            LogError(null, message);
        }

        /// <summary>
        /// Write an error message into the event log.
        /// </summary>
        /// <param name="message">The message.</param>
        public static void LogError(string category, string message)
        {
            if (category == null)
                AddLogLine(message, true);
            else
                AddLogLine(string.Format("[{0}] : {1}", category, message), true);
        }

        /// <summary>
        /// Write the entire exception string into the event log.
        /// </summary>
        /// <param name="ex">The exception.</param>
        public static void LogException(Exception ex)
        {
            LogException(null, ex);
        }

        /// <summary>
        /// Write the entire exception string into the event log.
        /// </summary>
        /// <param name="category">The category (like a project name).</param>
        /// <param name="ex">The exception.</param>
        public static void LogException(string category, Exception ex)
        {
            if (category == null)
            {
                AddLogLine(ex.ToString(), true);
            }
            else
            {
                AddLogLine(string.Format("[{0}] : {1}", category, ex.ToString()), true);
            }
        }

        /// <summary>
        /// Write a new line with the given message into the event log.
        /// </summary>
        /// <param name="logMessage">The log message.</param>
        /// <param name="isError">True for an error message.</param>
        private static void AddLogLine(string logMessage, bool isError)
        {
            EventLog log = new EventLog();
            log.Source = _logSource;
            //
            try
            {
                log.WriteEntry(logMessage, (isError ? EventLogEntryType.Error : EventLogEntryType.Information));
            }
            catch
            {
                log.Clear();
                log.WriteEntry(logMessage, (isError ? EventLogEntryType.Error : EventLogEntryType.Information));
            }
            log.Close();
        }
    }
}
