<?php
//
// +--------------------------------------------------------------+
// |     FIRST PERSON VIEW                                        |
// |     http://oscardias.com/php/php-class-first-person-view/    |
// +--------------------------------------------------------------+
//
//
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2, or (at your option)
//   any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; if not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//
/**
 * FIRST PERSON VIEW class
 *
 * PHP class that allows you to create a first person view website,
 * like a virtual tour or a street view. You have to provide a map with
 * coordinates of the available positions, and 4 images for each position.
 * Each image is represents one angle: 0, 90, 180 and 270.
 *
 * The X and Y plan and the angles will be something like this:
 *   5            ^
 *   4           90o
 *   3     < 180o   0o >
 *   2          270o
 *   1            v
 * X 0  1  2  3  4  5  6 ...
 *   Y
 *
 * So:
 * - if you are in the coordinates [1,2], with angle 0 and you move
 * forward, the algorithm will place you at [2,2] with angle 0.
 * - if you are in the coordinates [2,2], with angle 0 and you turn
 * right, the algorithm will place you at [2,2] with angle 270.
 * - if you are in the coordinates [2,2], with angle 270 and you move
 * back, the algorithm will place you at [2,3] with angle 270.
 * - and so on...
 *
 * You can hardcode the map coordinates with the images, in case you use
 * it for a small store virtual tour, or you can load this info
 * from a database. You can load dinamically only the coordinates you need.
 *
 * @author   Oscar de Souza Dias
 * @website  http://oscardias.com/php/php-class-first-person-view/
 * @version  1.0.0
 * @date     21 August 2010
 */
class firstPersonView {

    /*******************
     * CLASS VARIABLES *
     *******************/

    /*
     * Map matrix with images
     * It should be loaded with the X and Y coordinates and the angle, for example:
     * $map[0][0][0]   = "image000.jpg";
     * $map[0][0][90]  = "image0090.jpg";
     * $map[0][0][180] = "image00180.jpg";
     * $map[0][0][270] = "image00270.jpg";
     * // coordinates 1,0 doesn't exist
     * $map[2][0][0]   = "image200.jpg";
     * ...
     * $map[3][3][90]  = "image3390.jpg";
     * $map[3][3][180] = "image33180.jpg";
     * $map[3][3][270] = "image33270.jpg";
     *
     * This map could be something like this:
     * [0,3]    X       [2,3]   [3,3]
     * [0,2]    X       X       [3,2]
     * [0,1]    [1,1]   [2,1]   [3,1]
     * [0,0]    X       [2,0]   [3,0]
     */
    var $map;

    /*
     * Path where the images are
     */
    var $imagePath;

    /*
     * Current coordinates
     */
    var $currentX;
    var $currentY;

    /*
     * Current angle
     */
    var $currentAngle;

    /***************
     * CONSTRUCTOR *
     ***************/
    /*
     * CONSTRUCTOR Function
     *
     * Set the basic variables
     *
     * @param  array   $map    Images matrix
     * @param  int     $curX   X coordinate - 0 by default
     * @param  int     $curY   Y coordinate - 0 by default
     * @param  int     $curAng Current angle - 0 by default
     * @param  string  $path   Images path
     */
    function firstPersonView($map, $curX = 0, $curY = 0, $curAng = 0, $path = "") {
        $this->map = $map;
        $this->currentX = $curX;
        $this->currentY = $curY;
        $this->currentAngle = $curAng;
        
        // Add slash
        if (substr($path, strlen($path) - 1, 1) != "/")
            $path = $path."/";

        $this->imagePath = $path;
    }

    /********************
     * PUBLIC FUNCTIONS *
     ********************/

    /*
     * List of functions:
     * checkForward
     * goForward
     * checkBack
     * goBack
     * checkRight
     * goRight
     * checkLeft
     * goLeft
     * checkTurnRight
     * turnRight
     * checkTurnLeft
     * turnLeft
     * currentView
     */

    /*
     * Check Forward
     *
     * Check if it's possible to move forward according to the coordinates and angle
     *
     * @return  bool
     * @access  public
     */
    function checkForward() {
        if ($this->_checkMove($this->currentX + round(cos(deg2rad($this->currentAngle))),
                $this->currentY + round(sin(deg2rad($this->currentAngle))),
                $this->currentAngle))
            return true;
        else
            return false;
    }

    /*
     * Go Forward
     *
     * Move forward according to the coordinates and angle
     *
     * @access  public
     */
    function goForward() {
        if ($this->checkForward()) {
            $this->currentX = $this->currentX + round(cos(deg2rad($this->currentAngle)));
            $this->currentY = $this->currentY + round(sin(deg2rad($this->currentAngle)));
        }
    }

    /*
     * Check Back
     *
     * Check if it's possible to move backward according to the coordinates and angle
     *
     * @return  bool
     * @access  public
     */
    function checkBack() {
        if ($this->_checkMove($this->currentX - round(cos(deg2rad($this->currentAngle))),
                $this->currentY - round(sin(deg2rad($this->currentAngle))),
                $this->currentAngle))
            return true;
        else
            return false;
    }

    /*
     * Go Back
     *
     * Move backward according to the coordinates and angle
     *
     * @access  public
     */
    function goBack() {
        if ($this->checkBack()) {
            $this->currentX = $this->currentX - round(cos(deg2rad($this->currentAngle)));
            $this->currentY = $this->currentY - round(sin(deg2rad($this->currentAngle)));
        }
    }

    /*
     * Check Right
     *
     * Check if it's possible to move right according to the coordinates and angle
     *
     * @return  bool
     * @access  public
     */
    function checkRight() {
        if ($this->_checkMove($this->currentX + round(sin(deg2rad($this->currentAngle))),
                $this->currentY - round(cos(deg2rad($this->currentAngle))),
                $this->currentAngle))
            return true;
        else
            return false;
    }

    /*
     * Go Right
     *
     * Move right according to the coordinates and angle
     *
     * @access  public
     */
    function goRight() {
        if ($this->checkRight()) {
            $this->currentX = $this->currentX + round(sin(deg2rad($this->currentAngle)));
            $this->currentY = $this->currentY - round(cos(deg2rad($this->currentAngle)));
        }
    }

    /*
     * Check Left
     *
     * Check if it's possible to move left according to the coordinates and angle
     *
     * @return  bool
     * @access  public
     */
    function checkLeft() {
        if ($this->_checkMove($this->currentX - round(sin(deg2rad($this->currentAngle))),
                $this->currentY + round(cos(deg2rad($this->currentAngle))),
                $this->currentAngle))
            return true;
        else
            return false;
    }

    /*
     * Go Left
     *
     * Move left according to the coordinates and angle
     *
     * @access  public
     */
    function goLeft() {
        if ($this->checkLeft()) {
            $this->currentX = $this->currentX - round(sin(deg2rad($this->currentAngle)));
            $this->currentY = $this->currentY + round(cos(deg2rad($this->currentAngle)));
        }
    }

    /*
     * Check Turn Right
     *
     * Check if it's possible to turn right according to the coordinates and angle
     *
     * @return  bool
     * @access  public
     */
    function checkTurnRight() {
        if ($this->_checkMove($this->currentX, $this->currentY,
                ($this->currentAngle == 0)?270:$this->currentAngle-90))
            return true;
        else
            return false;
    }

    /*
     * Turn Right
     *
     * Turn Right according to the coordinates and angle
     *
     * @access  public
     */
    function turnRight() {
        if ($this->checkTurnRight()) {
            if ($this->currentAngle == 0)
                $this->currentAngle = 270;
            else
                $this->currentAngle -= 90;
        }
    }

    /*
     * Check Turn Left
     *
     * Check if it's possible to turn left according to the coordinates and angle
     *
     * @return  bool
     * @access  public
     */
    function checkTurnLeft() {
        if ($this->_checkMove($this->currentX, $this->currentY,
                ($this->currentAngle == 270)?0:$this->currentAngle+90))
            return true;
        else
            return false;
    }

    /*
     * Turn Left
     *
     * Turn Left according to the coordinates and angle
     *
     * @access  public
     */
    function turnLeft() {
        if ($this->checkTurnLeft()) {
            if ($this->currentAngle == 270)
                $this->currentAngle = 0;
            else
                $this->currentAngle += 90;
        }
    }

    /*
     * Current View
     *
     * Return current image filename
     *
     * @return      string
     * @access      public
     */
    function currentView() {
        echo $this->imagePath.$this->map[$this->currentX][$this->currentY][$this->currentAngle];
    }

    /********************
     * PRIVATE FUNCTIONS *
     ********************/

    /*
     * List of functions:
     * _checkMove
     */

    /*
     * Check Move
     *
     * Check if it's possible to change coordinate
     *
     * @return  bool
     * @param   int    $x   Target X coordinate
     * @param   int    $y   Target Y coordinate
     * @param   int    $a   Target angle
     * @access  private
     */
    function _checkMove($x, $y, $a) {
        if (isset($this->map[$x][$y][$a]))
            return true;
        else
            return false;
    }
}
