<?php
/**
 * This class can negotiate the use of cache by user-agent.
 * It uses "Expires" in response header and
 * reads "If-Modified-Since" in request header
 * to determine wheather the application could send
 * the 304 HTTP code (Not Modified).
 *
 * Benefits:
 * - decrease traffic of data.
 * - take the page load faster.
 * - does not affect the application behavior.
 * - easy of use.
 *
 * @version 1.0 2012-07-19
 * @author Rubens Takiguti Ribeiro <rubs33@gmail.com>
 * @license LGPL 3 <http://www.gnu.org/licenses/lgpl.html>
 * @copyright Copyright 2012 Rubens Takiguti Ribeiro
 */
final class AutoExpires {


/// Attributes


    /**
     * Cache time in seconds (default: no cache)
     * @var int
     */
    private static $cacheTime = 0;

    /**
     * Type of cache: "public" or "private"
     * @var string
     */
    private static $cacheType = 'private';

    /**
     * List of strategies to determine if the content is still valid.
     * Posible strategies are:
     * - 'if-modified': check if Hash of content was modified.
     * - 'if-expired': check if the time of cache was expired.
     * @var array[string]
     */
    private static $strategies = array(
        'if-modified',
        'if-expired'
    );

    /**
     * List of user-defined strategies to determine if the content is still valid.
     * It is used to avoid coupling of classes.
     * Each callback must have the signature:
     * bool callback(void)
     * It should return true if content is still valid, or false if it is not.
     * @var array[callback]
     */
    private static $callbacksStrategies = array();

    /**
     * Internal flag to control wheather the class was already started.
     * @var bool
     */
    private static $started = false;


/// Magic methods


    /**
     * Private constructor (use public static methods only)
     */
    private function __construct() {
        //void
    }


/// Public methods


    /**
     * Define time limit for cache (in seconds).
     * @param int $time Time in seconds
     * @return void
     */
    public static function setCacheTime($time) {
        if (self::$started) {
            trigger_error('AutoExpires was already started', E_USER_NOTICE);
            return;
        }
        self::$cacheTime = (int)$time;
    }

    /**
     * Returns the time limit for cache.
     * @return int
     */
    public static function getCacheTime() {
        return self::$cacheTime;
    }

    /**
     * Define cache type ('public' or 'private').
     * @param string $type One of 'public' or 'private'
     * @return void
     */
    public static function setCacheType($type) {
        if (self::$started) {
            trigger_error('AutoExpires was already started', E_USER_NOTICE);
            return;
        }
        switch ($type) {
        case 'public':
        case 'private':
            self::$cacheType = $type;
            break;
        }
    }

    /**
     * Returns the type of cache.
     * @return string
     */
    public static function getCacheType() {
        return self::$cacheType;
    }

    /**
     * Specify a list of strategies to determine if the content is still valid.
     * @param array[string] $strategis List of strategies
     * @return void
     */
    public static function setStrategies(array $strategies) {
        self::clearStrategies();
        foreach ($strategies as $strategy) {
            self::addStrategy($strategy);
        }
    }

    /**
     * Get the list of strategies to determine if the content is still valid.
     * @return array[string]
     */
    public static function getStrategies() {
        return self::$strategies;
    }

    /**
     * Clear the list of strategies
     * @return void
     */
    public static function clearStrategies() {
        self::$strategies = array();
    }

    /**
     * Add a strategy to the list of strategies to determine if the content is still valid.
     * @param string $strategy The strategy to add.
     * @return void
     */
    public static function addStrategy($strategy) {
        if (self::$started) {
            trigger_error('AutoExpires was already started', E_USER_NOTICE);
            return;
        }
        switch ($strategy) {
        case 'if-modified':
        case 'if-expired':
            self::$strategies[] = $strategy;
            break;
        }
    }

    /**
     * Check if the strategy is used.
     * @param string $strategy Strategy to check.
     * @return bool
     */
    public static function usesStrategy($strategy) {
        return in_array($strategy, self::getStrategies());
    }

    /**
     * Define a list of callback to use as strategy to determine if the content is still valid.
     * @see #callbacksStrategies
     * @param callback $callback Callback with signature "bool callback(void)"
     * @return void
     */
    public static function setCallbacksStrategies(array $callbacks) {
        self::clearCallbacksStrategies();
        foreach ($callbacks as $callback) {
            self::addCallbackStrategy($callback);
        }
    }

    /**
     * Return the list of callback to use as strategy to determine if the content is still valid.
     * @see #callbacksStrategies
     * @return array[callback]
     */
    public static function getCallbacksStrategies() {
        return self::$callbacksStrategies;
    }

    /**
     * Clear the list of callback.
     * @see #callbacksStrategies
     * @return void
     */
    public static function clearCallbacksStrategies() {
        self::$callbacksStrategies = array();
    }

    /**
     * Add a callback that implements a strategy to determine if the content is still valid.
     * @see #callbacksStrategies
     * @param callback $callback Callback with signature "bool callback(void)"
     * @return void
     */
    public static function addCallbackStrategy(callbak $callback) {
        if (self::$started) {
            trigger_error('AutoExpires was already started', E_USER_NOTICE);
            return;
        }
        self::$callbacksStrategies[] = $callback;
    }

    /**
     * Starts the class. It should be called before printing anything.
     * @return void
     */
    public static function start() {
        if (self::$started) {
            trigger_error('AutoExpires was already started', E_USER_NOTICE);
            return;
        }
        if (headers_sent($file, $line)) {
            $msg = sprintf(
                'Headers already sent in %s on line %d',
                $file,
                $line
            );
            trigger_error($msg, E_USER_NOTICE);
        }

        self::$started = true;

        ob_start(array(__CLASS__, 'finish'));
    }

    /**
     * Try to get If-Modified-Since directive from user-agent.
     * @return int | bool
     */
    public static function getUserAgentCacheTime() {
        $date = false;

        // Check in $_SERVER
        if (array_key_exists('HTTP_IF_MODIFIED_SINCE', $_SERVER)) {
            $date = $_SERVER['HTTP_IF_MODIFIED_SINCE'];

        // Check in request header
        } else {
            $headers = array();
            if (function_exists('getallheaders')) {
                $headers = getallheaders();
            } elseif (function_exists('apache_request_headers')) {
                $headers = apache_request_headers();
            } elseif (function_exists('http_get_request_headers')) {
                $headers = http_get_request_headers();
            }
            foreach ($headers as $key => $value) {
                if (strcasecmp('If-Modified-Since', $key) == 0) {
                    $date = $value;
                    break;
                }
            }
        }

        // Not found
        if (!$date) {
            return false;
        }

        // Format as timestamp
        $d = strptime($date, '%a, %d %b %Y %T');
        if (!$d) {
            return false;
        }

        return gmmktime(
            $d['tm_hour'], $d['tm_min'], $d['tm_sec'],
            $d['tm_mon'] + 1, $d['tm_mday'], 1900 + $d['tm_year']
        );
    }


/// Reserved method


    /**
     * YOU DO NOT NEED TO CALL THIS METHOD.
     * Method called by ob_start callback.
     * It checks wheather the HTTP 304 code should be sent.
     * @param string $content: content from ob.
     * @return string
     */
    public static function finish($content) {
        $hash = self::calculateHash($content);

        if (self::userAgentHasValidCache($hash)) {
            self::sendNotModifiedHeader();
            return '';
        } else {
            self::setSessionHash($hash);
            self::sendExpireHeader();
            return $content;
        }
    }


/// Private methods


    /**
     * Send 304 HTTP code to user-agent.
     * @return void
     */
    private static function sendNotModifiedHeader() {

        // Prevent user-defined locale
        $defaultLocale = setlocale(LC_TIME, '0');
        setlocale(LC_TIME, 'C');

        header('HTTP/1.0 304 Not Modified');
        header('Date: ' . gmstrftime('%a, %d %b %Y %T %Z', $_SERVER['REQUEST_TIME']));

        // Remove Cache-Control, Pragma and Expires defined by session.cache_limiter
        header('Cache-Control: ');
        header('Pragma: ');
        header('Expires: ');

        // Reset user-defined locale
        setlocale(LC_TIME, $defaultLocale);
    }

    /**
     * Send HTTP header with Expire header.
     * @return void
     */
    private static function sendExpireHeader() {

        // Prevent user-defined locale
        $defaultLocale = setlocale(LC_TIME, '0');
        setlocale(LC_TIME, 'C');

        header('Cache-Control: ' . self::getCacheType());
        header('Pragma: ');
        header('Date: ' . gmstrftime('%a, %d %b %Y %T %Z', $_SERVER['REQUEST_TIME']));
        header('Expires: ' . gmstrftime('%a, %d %b %Y %T %Z', $_SERVER['REQUEST_TIME'] + self::getCacheTime()));
        header('Last-Modified: ' . gmstrftime('%a, %d %b %Y %T %Z', $_SERVER['REQUEST_TIME']));

        // Reset user-defined locale
        setlocale(LC_TIME, $defaultLocale);
    }

    /**
     * Calculates the hash of a content.
     * @param string $content Original content.
     * @return string
     */
    private static function calculateHash($content) {
        return number_format(crc32($content), 0, '', '');
    }

    /**
     * Checks wheather the user-agent has informed If-Modified-Sice directive,
     * and uses the list of strategies to determine if user-agent cache is
     * still valid.
     * @param string $hash
     * @return bool
     */
    private static function userAgentHasValidCache($hash) {

        // Try to get If-Modified-Since value
        $userAgentCacheTime = self::getUserAgentCacheTime();

        if (!$userAgentCacheTime) {
            return false;
        }

        // Hash changed
        if (self::usesStrategy('if-modified')) {
            $sessionHash = self::getSessionHash();
            if (!$sessionHash || $sessionHash != $hash) {
                return false;
            }
        }

        // Time expired
        if (self::usesStrategy('if-expired')) {
            if ($userAgentCacheTime + self::getCacheTime() < $_SERVER['REQUEST_TIME']) {
                return false;
            }
        }

        // User-defined strategies
        foreach (self::getCallbacksStrategies() as $callback) {
            if (!call_user_func($callback)) {
                return false;
            }
        }

        // Valid cache
        return true;
    }


/// Session control methods


    /**
     * Save hash in session
     * @param string $hash Hash to save
     * @return void
     */
    private static function setSessionHash($hash) {
        session_start();
        $_SESSION[__FILE__][$_SERVER['REQUEST_URI']] = $hash;
    }

    /**
     * Get hash from session or false if it was not defined.
     * @return string || false
     */
    private static function getSessionHash() {
        session_start();
        if (!isset($_SESSION[__FILE__][$_SERVER['REQUEST_URI']])) {
            return false;
        }
        return $_SESSION[__FILE__][$_SERVER['REQUEST_URI']];
    }
}