﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Drawing;

namespace CssSpriteGenerator
{
    public class SizeUtils
    {
        /// <summary>
        /// Determines whether the given Size has both unknown width and height
        /// </summary>
        /// <param name="size"></param>
        /// <returns></returns>
        public static bool IsEmptySize(Size size)
        {
            return ((size.Width < 0) && (size.Height < 0));
        }

        /// <summary>
        /// Returns true if both width and height of the given Size are set
        /// </summary>
        /// <param name="size"></param>
        /// <returns></returns>
        public static bool IsCompleteSize(Size size)
        {
            return ((size.Width >= 0) && (size.Height >= 0));
        }

        /// <summary>
        /// Takes a Size that is possibly missing a width or height, and returns a Size
        /// with both width and height filled in.
        /// 
        /// If both width and height are missing, than a Size with both width and height set to -1 
        /// is returned.
        /// </summary>
        /// <param name="incompleteSize">
        /// Possibly incomplete Size.
        /// </param>
        /// <param name="aspectRatio">
        /// Aspect ratio of the image. Can be used to derive one dimension from the other.
        /// 
        /// Aspect ratio is width / height.
        /// </param>
        /// <returns></returns>
        public static Size CompletedSize(Size incompleteSize, float aspectRatio)
        {
            if (IsEmptySize(incompleteSize)) { return new Size(-1, -1); }

            if (IsCompleteSize(incompleteSize)) { return incompleteSize; }

            // At this stage, precisely one of the dimensions is not given

            if (incompleteSize.Width >= 0)
            {
                return new Size(incompleteSize.Width, (int)Math.Floor(incompleteSize.Width / aspectRatio));
            }

            return new Size((int)Math.Floor(incompleteSize.Height * aspectRatio), incompleteSize.Height);
        }

        /// <summary>
        /// Returns the area of a Size.
        /// </summary>
        /// <param name="size"></param>
        /// <returns></returns>
        public static long Area(Size size)
        {
            return (long)size.Width * size.Height;
        }

    }
}
