﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;

using System.Security.Cryptography;
using System.Text.RegularExpressions;
using System.IO;

/// <summary>
/// Helper Class for using the Google Mailhide API
/// </summary>
public class Mailhide : IDisposable
{
    protected SymmetricAlgorithm cryptoProvider = null;

    protected string privateKey = string.Empty;

	/// <summary>
    /// Initializes a default instance of the Mailhide class.
	/// </summary>
    public Mailhide()
	{
        Init();
	}

    /// <summary>
    /// Initializes a new instance of the Mailhide class with the specified private key.
    /// </summary>
    /// <param name="privateKey">A 32 character hexadecimal string</param>
    public Mailhide(string privateKey)
    {
        Init();

        PrivateKey = privateKey;
    }

    /// <summary>
    /// Initialize the crypto provider
    /// </summary>
    private void Init()
    {
        cryptoProvider = new AesManaged();
        cryptoProvider.Mode = CipherMode.CBC;       // The default mode
        cryptoProvider.IV = new byte[16];           // 16 null bytes
        cryptoProvider.Padding = PaddingMode.None;  // Use custom padding in the code below
    }

    /// <summary>
    /// A 32 character hexadecimal string
    /// </summary>
    public string PrivateKey
    {
        get 
        {
            return privateKey; 
        }
        set
        {
            if (string.IsNullOrEmpty(value))
            {
                throw new ArgumentNullException("PrivateKey");
            }

            //  Beginning of string
            //  Any character in this class: [0-9a-fA-F], exactly 32 repetitions
            //  End of string
            Regex regex = new Regex("^[0-9a-fA-F]{32}$");
            if (!regex.IsMatch(value))
            {
                throw new ArgumentOutOfRangeException("PrivateKey", "PrivateKey must be a 32 character hexadecimal string");
            }

            privateKey = value;

            // The Private Key consists of 2 digit hexadecimal characters.
            // Convert the string into a byte array
            byte[] key = new byte[privateKey.Length / 2];
            for (int i = 0; i < key.Length; i++)
            {
                key[i] = Convert.ToByte(privateKey.Substring(i * 2, 2), 16);
            }

            cryptoProvider.Key = key; // Set the encryption key
        }
    }

    /// <summary>
    /// A byte array of the encrypted email address.
    /// </summary>
    public byte[] EncryptedData { get; protected set; }
    
    /// <summary>
    /// The encrypted and encoded url-safe email address
    /// </summary>
    public string EncryptedEmail { get; protected set; }

    /// <summary>
    /// Encrypts an email address
    /// </summary>
    /// <param name="emailAddress">The plain text email address</param>
    /// <returns>An encrypted and encoded url-safe email address</returns>
    public string EncryptEmail(string emailAddress)
    {
        if (string.IsNullOrEmpty(emailAddress))
        {
            throw new ArgumentNullException("emailAddress");
        }
        if (string.IsNullOrEmpty(PrivateKey))
        {
            throw new ArgumentNullException("PrivateKey");
        }

        // Pad the email address as necessary to a 16 bit block size as required by AES
        string paddedEmailAddress = PadString(emailAddress, 16);

        // Encrypt the padded email address
        EncryptedData = Encrypt(paddedEmailAddress);

        // Encode the encrypted email address and make it url-safe.
        // This is the encrypted email that should be used in the Mailhide querystring.
        EncryptedEmail = Convert.ToBase64String(EncryptedData).Replace("+", "-").Replace("/", "_");

        return EncryptedEmail;
    }

    /// <summary>
    /// Pads the input string to a fixed block size as required by AES.
    /// </summary>
    /// <param name="inputString">The string to pad</param>
    /// <param name="blockSize">The block size to use. Should be 16 for AES.</param>
    /// <returns>The padded string</returns>
    protected string PadString(string inputString, int blockSize)
    {
        if (string.IsNullOrEmpty(inputString))
        {
            throw new ArgumentNullException("inputString");
        }
        if (blockSize <= 0)
        {
            throw new ArgumentOutOfRangeException("blockSize", "Must be a positive integer");
        }

        string paddedString = string.Empty;

        // Pad the string to a fixed block size.
        // For example, if the block size is 16:
        // Pad a 10 character string with 6 chars (1 * blocksize)
        // Pad a 30 character string with 2 chars (2 * blocksize)
        // Use the number of characters to pad with as the padding character.

        int numToPad = blockSize - (inputString.Length % blockSize);
        string padChars = new string((char)numToPad, numToPad);
        paddedString = inputString + padChars;
        return paddedString;
    }

    /// <summary>
    /// Encrypts the plain text string
    /// </summary>
    /// <param name="plainText">The string to encrypt</param>
    /// <returns>The encrypted string as a byte array</returns>
    protected byte[] Encrypt(string plainText)
    {
        byte[] encryptedData = null;

        // Key size is expressed in bits, so multiply the
        // key length * 8 to get the key size
        if (!cryptoProvider.ValidKeySize(cryptoProvider.Key.Length * 8))
        {
            throw new Exception(string.Format("Invalid Key Size: {0}", cryptoProvider.Key.Length * 8));
        }

        // Create an encryptor to perform the stream transform.
        ICryptoTransform encryptor = cryptoProvider.CreateEncryptor();

        // Create the streams used for encryption.
        using (MemoryStream memoryStream = new MemoryStream())
        {
            using (CryptoStream cryptoStream = new CryptoStream(memoryStream, encryptor, CryptoStreamMode.Write))
            {
                using (StreamWriter streamWriter = new StreamWriter(cryptoStream))
                {
                    // Write all data to the stream.
                    streamWriter.Write(plainText);
                }

                encryptedData = memoryStream.ToArray();
            }
        }

        return encryptedData;
    }

    /// <summary>
    /// Zeros the crypto provider's in-memory data and releases all resources.
    /// Call this method from within a using clause after encryption has been completed.
    /// </summary>
    public void Clear()
    {
        cryptoProvider.Clear();
    }

    /// <summary>
    /// Releases the resources used by the crypto provider.
    /// </summary>
    public void Dispose()
    {
        if (cryptoProvider != null)
        {
            cryptoProvider.Dispose();
            cryptoProvider = null;
        }
    }
}