﻿// -- FILE ------------------------------------------------------------------
// name       : PersistentUserForm.cs
// project    : Itenso Web User Forms
// created    : Jani Giannoudis - 2008.10.30
// language   : c#
// environment: .NET 2.0
// copyright  : (c) 2008-2012 by Itenso GmbH, Switzerland
// --------------------------------------------------------------------------
using System;
using System.IO;
using System.Web.UI;
using System.Web.UI.WebControls;
using Itenso.WebUserForms.Controls;
using Itenso.WebUserForms.Data.Form;
using Itenso.WebUserForms.Runtime;

// --------------------------------------------------------------------------
public partial class PersistentPage : Page 
{

	// ------------------------------------------------------------------------
	protected override void OnInit( EventArgs e )
	{
		base.OnInit( e );

		MasterPage masterPage = Master as MasterPage;
		if ( masterPage != null )
		{
			masterPage.ShowRuntimeHeaderInfo = true;
		}
	} // OnInit

	// ------------------------------------------------------------------------
	protected override void OnLoad( EventArgs e )
	{
		// load form control
		this.userForm = new UserFormLoader( "~/UserForms/PersistentUserForm.ascx" ).Load();

		this.commandManager = new UserFormCommandManager( this.userForm );
		this.commandManager.Command += FormCommand;

		FormPlaceHolder.Controls.Add( this.userForm );
		if ( !Page.IsPostBack )
		{
			LoadFormData();
		}

		base.OnLoad( e );
	} // OnLoad

	// ------------------------------------------------------------------------
	private void UpdateCommands( IForm form )
	{
		this.commandManager.EnableCommand( commandFormSave, !form.IsLocked );
		this.commandManager.EnableCommand( commandFormLock, !form.IsLocked );
		this.commandManager.EnableCommand( commandFormUnlock, form.IsLocked );
	} // UpdateCommands

	// ------------------------------------------------------------------------
	private void LoadFormData()
	{
		// file check
		string fileName = MapPath( virtualFileName );
		if ( !File.Exists( fileName ) )
		{
			return;
		}

		// load form data
		using ( StreamReader streamReader = new StreamReader( fileName ) )
		{
			IForm form = FormXml.Instance.Load( streamReader );
			UserFormAdapter.ApplyForm( this.userForm, form );
			UpdateCommands( form );
		}
	} // LoadFormData

	// ------------------------------------------------------------------------
	private void SaveFormData()
	{
		IForm form = UserFormAdapter.ExtractForm( this.userForm );

		string fileName = MapPath( virtualFileName );
		if ( File.Exists( fileName ) ) // update existing form
		{
			File.Delete( fileName );
			form.MarkUpdated( DateTime.Now, "DemoUser" );
		}
		else // new form
		{
			form.FormId = "1";
			form.SetCreated( DateTime.Now, "DemoUser" );
		}

		// ensure directory
		string directory = new FileInfo( fileName ).DirectoryName;
		if ( !string.IsNullOrEmpty( directory ) && !Directory.Exists( directory ) )
		{
			Directory.CreateDirectory( directory );
		}

		// save form data
		using ( StreamWriter streamWriter = new StreamWriter( fileName ) )
		{
			FormXml.Instance.Save( form, streamWriter );
			UserFormAdapter.ApplyForm( this.userForm, form );
		}

		UpdateCommands( form );
	} // SaveFormData

	// ------------------------------------------------------------------------
	private void ChangeFormLock( bool isLocked )
	{
		IForm form = UserFormAdapter.ExtractForm( this.userForm );
		form.IsLocked = isLocked;
		UpdateCommands( form );
	} // ChangeFormLock

	// ------------------------------------------------------------------------
	private void FormCommand( object sender, CommandEventArgs e )
	{
		switch ( e.CommandName.ToLower() )
		{
			case commandFormSave:
				SaveFormData();
				break;
			case commandFormLoad:
				LoadFormData();
				break;
			case commandFormLock:
				ChangeFormLock( true );
				break;
			case commandFormUnlock:
				ChangeFormLock( false );
				break;
		}
	} // FormCommand

	// ------------------------------------------------------------------------
	// members
	private UserControl userForm;
	private UserFormCommandManager commandManager;

	private const string virtualFileName = "~/Data/PersistentUserForm.xml";

	private const string commandFormSave = "formsave";
	private const string commandFormLoad = "formload";
	private const string commandFormLock = "formlock";
	private const string commandFormUnlock = "formunlock";

} // class PersistentPage
// -- EOF -------------------------------------------------------------------
