﻿// -- FILE ------------------------------------------------------------------
// name       : UserFormAdapter.cs
// project    : Itenso Web User Forms
// created    : Jani Giannoudis - 2008.10.30
// language   : c#
// environment: .NET 2.0
// copyright  : (c) 2008-2012 by Itenso GmbH, Switzerland
// --------------------------------------------------------------------------
using System;
using System.Web.UI;
using Itenso.WebUserForms.Data.Form;
using Itenso.WebUserForms.Controls;

namespace Itenso.WebUserForms.Runtime
{

	// ------------------------------------------------------------------------
	public static class UserFormAdapter
	{

		// ----------------------------------------------------------------------
		public static IForm ExtractForm( UserControl userControl )
		{
			if ( userControl == null )
			{
				throw new ArgumentNullException( "userControl" );
			}

			IUserForm userForm = UserFormCollector.CollectUserForm( userControl );
			return ExtractForm( userForm );
		} // ExtractForm

		// ----------------------------------------------------------------------
		public static void ApplyForm( UserControl userControl, IForm form )
		{
			ApplyForm( userControl, form, false );
		} // ApplyForm

		// ----------------------------------------------------------------------
		public static void ApplyForm( UserControl userControl, IForm form, bool versionCheck )
		{
			if ( userControl == null )
			{
				throw new ArgumentNullException( "userControl" );
			}
			if ( form == null )
			{
				throw new ArgumentNullException( "form" );
			}

			IUserForm userForm = UserFormCollector.CollectUserForm( userControl );
			ApplyForm( userForm, form, versionCheck );
		} // ApplyForm

		// ----------------------------------------------------------------------
		private static void ApplyForm( IUserForm userForm, IForm form, bool versionCheck )
		{
			if ( userForm == null )
			{
				throw new ArgumentNullException( "userForm" );
			}
			if ( form == null )
			{
				throw new ArgumentNullException( "form" );
			}
			if ( !form.FormType.Equals( userForm.Header.Type ) )
			{
				throw new UserFormAdapterException( "incompatible user form name", userForm, form );
			}
			if ( !form.Name.Equals( userForm.Header.Name ) )
			{
				throw new UserFormAdapterException( "incompatible user form name", userForm, form );
			}
			if ( versionCheck )
			{
				if ( !form.FormVersion.Equals( userForm.Header.FormVersion ) )
				{
					throw new UserFormAdapterException( "incompatible user form version", userForm, form );
				}
			}

			// header
			userForm.Header.FormType = form.FormType;
			userForm.Header.FormName = form.Name;
			userForm.Header.FormId = form.FormId;
			userForm.Header.FormVersion = form.FormVersion;
			userForm.Header.FormCreated = form.Created;
			userForm.Header.FormCreatedByUser = form.CreatedByUser;
			userForm.Header.FormLastUpdated = form.LastUpdated;
			userForm.Header.FormLastUpdatedByUser = form.LastUpdatedByUser;

			ApplyFormItems( userForm, form );
		} // ApplyForm

		// ----------------------------------------------------------------------
		private static void ApplyFormItems( IUserForm userForm, IFormGroup formGroup )
		{
			if ( !formGroup.Name.Equals( userForm.Header.Name ) )
			{
				return;
			}

			foreach ( IFormEntity formEntity in formGroup.Entities )
			{
				switch ( formEntity.Kind )
				{
					case FormEntityType.Form:
						IFormGroup subFormGroup = formEntity as IFormGroup;
						if ( subFormGroup != null )
						{
							IUserForm webSubForm = userForm.Subforms.FindByFormName( subFormGroup.Name );
							if ( webSubForm != null )
							{
								ApplyFormItems( webSubForm, subFormGroup );
							}
						}
						break;
					case FormEntityType.Field:
						IFormField formField = formEntity as IFormField;
						if ( formField != null )
						{
							IUserFormField userFormField = userForm.Fields.FindByFieldName( formField.Name );
							if ( userFormField != null )
							{
								userFormField.FieldValue = formField.Content;
							}
						}
						break;
				}
			}
		} // ApplyFormItems

		// ----------------------------------------------------------------------
		private static IForm ExtractForm( IUserForm userForm )
		{
			if ( userForm == null )
			{
				throw new ArgumentNullException( "userForm" );
			}

			Form form = new Form();
			form.FormType = userForm.Header.Type;
			form.Name = userForm.Header.Name;
			form.FormVersion = userForm.Header.FormVersion;
			form.FormId = userForm.Header.FormId;
			if ( userForm.Header.FormCreated.HasValue )
			{
				form.Created = userForm.Header.FormCreated.Value;
			}
			form.CreatedByUser = userForm.Header.FormCreatedByUser;
			if ( userForm.Header.FormLastUpdated.HasValue )
			{
				form.LastUpdated = userForm.Header.FormLastUpdated.Value;
			}
			form.LastUpdatedByUser = userForm.Header.FormLastUpdatedByUser;

			ExtractFormItems( userForm, form );

			return form;
		} // ExtractForm

		// ----------------------------------------------------------------------
		private static void ExtractFormItems( IUserForm userForm, IForm form )
		{
			foreach ( IUserFormField userFormField in userForm.Fields )
			{
				FormField formField = new FormField( userFormField.FieldName, userFormField.FieldValue );
				form.Entities.Add( formField );
			}

			foreach ( IUserForm webSubform in userForm.Subforms )
			{
				Form subform = new Form( webSubform.Header.Name );
				form.Entities.Add( subform );
				ExtractFormItems( webSubform, subform );
			}
		} // ExtractFormItems

	} // class UserFormAdapter

} // namespace Itenso.WebUserForms.Runtime
// -- EOF -------------------------------------------------------------------
