// -- FILE ------------------------------------------------------------------
// name       : XmlSchemaBase.cs
// project    : Itenso Web User Forms
// created    : Jani Giannoudis - 2008.10.30
// language   : c#
// environment: .NET 2.0
// copyright  : (c) 2008-2012 by Itenso GmbH, Switzerland
// --------------------------------------------------------------------------
using System;
using System.IO;
using System.Xml;
using System.Xml.Schema;
using System.Diagnostics;
using System.Globalization;

namespace Itenso.WebUserForms.Data
{

	// ------------------------------------------------------------------------
	public abstract class XmlSchemaBase
	{

		// ----------------------------------------------------------------------
		public const string XsdPrefix = "xsd";
		public const string XsdSuffix = ".xsd";

		// ----------------------------------------------------------------------
		public static XmlSchema LoadSchema( Type type )
		{
			return LoadSchema( type, type.Name + XsdSuffix );
		} // LoadSchema

		// ----------------------------------------------------------------------
		public static XmlSchema LoadSchema( Type type, string resourceName )
		{
			Stream schemaStream = ResourceTool.GetResourceAsStream( type, resourceName );
			return XmlSchema.Read( schemaStream, XmlSchemaValidationHandler );
		} // LoadSchema

		// ----------------------------------------------------------------------
		public static XmlReaderSettings CreateStrictSchemaValidationSettings( params XmlSchema[] schemas )
		{
			XmlReaderSettings settings = new XmlReaderSettings();
			if ( schemas != null )
			{
				for ( int i = 0; i < schemas.Length; i++ )
				{
					XmlSchema schema = schemas[ i ];
					if ( schema != null )
					{
						settings.Schemas.Add( schema );
					}
				}
			}
			settings.ValidationFlags =
				//XmlSchemaValidationFlags.ProcessInlineSchema |
				//XmlSchemaValidationFlags.ProcessSchemaLocation |
				XmlSchemaValidationFlags.ProcessIdentityConstraints |
				XmlSchemaValidationFlags.ReportValidationWarnings;
			settings.ValidationType = ValidationType.Schema;
			settings.ValidationEventHandler += XmlSchemaValidationWarningsAsExceptionsHandler;
			settings.ValidationEventHandler += XmlSchemaValidationErrorsAsExceptionsHandler;
			settings.ConformanceLevel = ConformanceLevel.Fragment;
			return settings;
		} // CreateStrictSchemaValidationSettings

		// ----------------------------------------------------------------------
		public static void XmlSchemaValidationHandler( object sender, ValidationEventArgs e )
		{
			if ( e.Severity == XmlSeverityType.Warning )
			{
				Debug.Fail( "XML Schema validation warning", e.Message );
			}
			else if ( e.Severity == XmlSeverityType.Error )
			{
				Debug.Fail( "XML Schema validation error", e.Message );
			}
		} // XmlSchemaValidationHandler

		// ----------------------------------------------------------------------
		public static void XmlSchemaValidationWarningsAsExceptionsHandler( object sender, ValidationEventArgs e )
		{
			if ( e.Severity == XmlSeverityType.Warning )
			{
				ThrowXmlExceptionFor( "XML Schema validation warning: {0}", e.Message, e.Exception );
			}
		} // XmlSchemaValidationWarningsAsExceptionsHandler

		// ----------------------------------------------------------------------
		public static void XmlSchemaValidationErrorsAsExceptionsHandler( object sender, ValidationEventArgs e )
		{
			if ( e.Severity == XmlSeverityType.Error )
			{
				ThrowXmlExceptionFor( "XML Schema validation error: {0}", e.Message, e.Exception );
			}
		} // XmlSchemaValidationErrorsAsExceptionsHandler

		// ----------------------------------------------------------------------
		private static void ThrowXmlExceptionFor( string message, string error, Exception e )
		{
			string exceptionText = String.Format( CultureInfo.InvariantCulture, message, error );
			XmlException xmlExc = e as XmlException;
			XmlSchemaException schemaExc = e as XmlSchemaException;
			if ( xmlExc != null )
			{
				throw new XmlException( exceptionText, e, xmlExc.LineNumber, xmlExc.LinePosition );
			}
			
			if ( schemaExc != null )
			{
				throw new XmlException( exceptionText, e, schemaExc.LineNumber, schemaExc.LinePosition );
			}
			
			if ( e != null )
			{
				throw new XmlException( exceptionText, e );
			}
			
			throw new XmlException( exceptionText );
		} // ThrowXmlExceptionFor

	} // class XmlSchemaBase

} // namespace Itenso.WebUserForms.Data
// -- EOF -------------------------------------------------------------------
