// -- FILE ------------------------------------------------------------------
// name       : ResourceTool.cs
// project    : Itenso Web User Forms
// created    : Jani Giannoudis - 2008.10.30
// language   : c#
// environment: .NET 2.0
// copyright  : (c) 2008-2012 by Itenso GmbH, Switzerland
// --------------------------------------------------------------------------
using System;
using System.IO;
using System.Reflection;

namespace Itenso.WebUserForms.Data
{

	// ------------------------------------------------------------------------
	internal static class ResourceTool
	{

		// ----------------------------------------------------------------------
		/// <summary>
		/// Opens a stream to the given embedded resource in the test assembly. Fails if the
		/// given test resource cannot be located in the assembly.
		/// </summary>
		/// <param name="type">the type of the embedded resource</param>
		/// <param name="name">the name of the embedded resource without assembly prefix</param>
		/// <returns>an open stream to the desired embedded resource</returns>
		/// <exception cref="EmbeddedResourceNotFoundException">if the resource could not be found</exception>
		public static Stream GetResourceAsStream( Type type, string name )
		{
			Assembly assembly = type.Assembly;
			string namespaceName = type.Namespace;
			string fullName = namespaceName + "." + name.Replace( '\\', '.' ).Replace( '/', '.' );
			Stream stream = assembly.GetManifestResourceStream( fullName );
			if ( stream == null )
			{
				throw new EmbeddedResourceNotFoundException( fullName );
			}
			return stream;
		} // GetResourceAsStream

		// ----------------------------------------------------------------------
		/// <summary>
		/// Searches for all embedded resources which are located in a 'directory' with
		/// the current class' name in the executing assembly. Like that every test
		/// class can have its own test data resources in a directory corresponding to
		/// its name. The returned names in the array can be used as an argument to the
		/// <see cref="OpenTestResourceStream"/> method.
		/// </summary>
		public static string[] GetTestResourceNames( Type type )
		{
			Assembly assembly = type.Assembly;
			string namespaceName = type.Namespace;
			string assemblyPrefix = namespaceName + "." + type.Name + ".";
			string[] resources = assembly.GetManifestResourceNames();
			int valid = 0;
			for ( int i = 0; i < resources.Length; i++ )
			{
				string resource = resources[ i ];
				if ( resource.StartsWith( assemblyPrefix ) )
				{
					valid++;
				}
				else
				{
					resources[ i ] = null;
				}
			}
			string[] testResources = new string[ valid ];
			valid = 0;
			for ( int i = 0; i < resources.Length; i++ )
			{
				string resource = resources[ i ];
				if ( resource != null )
				{
					testResources[ valid ] = resource.Substring( assemblyPrefix.Length );
					valid++;
				}
			}
			return testResources;
		} // GetTestResourceNames

		// ----------------------------------------------------------------------
		/// <summary>
		/// Opens a stream to the given embedded resource in the test assembly. Fails if the
		/// given test resource cannot be located in the assembly.
		/// </summary>
		/// <param name="type">the type of the embedded resource</param>
		/// <param name="name">the name of the embedded resource without assembly prefix</param>
		/// <returns>an open stream to the desired embedded resource</returns>
		/// <seealso cref="GetTestResourceNames"/>
		public static Stream OpenTestResourceStream( Type type, string name )
		{
			Assembly assembly = type.Assembly;
			string namespaceName = type.Namespace;
			string fullName = namespaceName + "." + type.Name + "." +
				name.Replace( '\\', '.' ).Replace( '/', '.' );
			Stream stream = assembly.GetManifestResourceStream( fullName );
			return stream;
		} // OpenTestResourceStream

	} // class ResourceTool

} // namespace Itenso.WebUserForms.Data
// -- EOF -------------------------------------------------------------------
