// -- FILE ------------------------------------------------------------------
// name       : LookupValueCollection.cs
// project    : Itenso Web User Forms
// created    : Jani Giannoudis - 2008.10.30
// language   : c#
// environment: .NET 2.0
// copyright  : (c) 2008-2012 by Itenso GmbH, Switzerland
// --------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;

namespace Itenso.WebUserForms.Data.Lookup
{

	// ------------------------------------------------------------------------
	public sealed class LookupValueCollection : ILookupValueCollection
	{

		// ----------------------------------------------------------------------
		/// <summary>
		/// Creates a new empty instance.
		/// </summary>
		public LookupValueCollection( string name )
		{
			if ( string.IsNullOrEmpty( name ) )
			{
				throw new ArgumentException( "name" );
			}

			this.name = name;
		} // LookupValueCollection

		// ----------------------------------------------------------------------
		/// <summary>
		/// Creates a new instance with all the values of the given collection.
		/// </summary>
		/// <param name="name">the lookup name.</param>
		/// <param name="lookupValue">the lookup values to add. may not be null.</param>
		public LookupValueCollection( string name, IEnumerable<ILookupValue> lookupValue )
			: this( name )
		{
			AddAll( lookupValue );
		} // LookupValueCollection

		// ----------------------------------------------------------------------
		public string Name
		{
			get { return this.name; }
		} // Name

		// ----------------------------------------------------------------------
		public int Count
		{
			get { return this.values.Count; }
		} // Count

		// ----------------------------------------------------------------------
		public ILookupValue this[ int index ]
		{
			get { return this.values[ index ]; }
		} // this[int]

		// ----------------------------------------------------------------------
		public ILookupValue this[ string key ]
		{
			get 
			{
				if ( string.IsNullOrEmpty( key ) )
				{
					throw new ArgumentNullException( "key" );
				}
				foreach ( ILookupValue value in this )
				{
					if ( key.Equals( value.Key ) )
					{
						return value;
					}
				}
				return null;
			}
		} // this[string]

		// ----------------------------------------------------------------------
		IEnumerator IEnumerable.GetEnumerator()
		{
			return values.GetEnumerator();
		} // IEnumerable.GetEnumerator

		// ----------------------------------------------------------------------
		public IEnumerator<ILookupValue> GetEnumerator()
		{
			return this.values.GetEnumerator();
		} // GetEnumerator

		// ----------------------------------------------------------------------
		public int IndexOf( ILookupValue value )
		{
			return this.values.IndexOf( value );
		} // IndexOf

		// ----------------------------------------------------------------------
		public int Add( ILookupValue lookupValue )
		{
			if ( lookupValue == null )
			{
				throw new ArgumentNullException( "lookupValue" );
			}
			int insertPos = values.Count;
			this.values.Add( lookupValue );
			return insertPos;
		} // Add

		// ----------------------------------------------------------------------
		public void Add( ILookupValue lookupValue, int pos )
		{
			if ( lookupValue == null )
			{
				throw new ArgumentNullException( "lookupValue" );
			}
			this.values.Insert( pos, lookupValue );
		} // Add

		// ----------------------------------------------------------------------
		public void AddAll( IEnumerable<ILookupValue> lookupValues )
		{
			if ( lookupValues == null )
			{
				throw new ArgumentNullException( "lookupValues" );
			}
			foreach ( ILookupValue lookupValue in lookupValues )
			{
				Add( lookupValue );
			}
		} // AddAll

		// ----------------------------------------------------------------------
		public void Remove( ILookupValue lookupValue )
		{
			this.values.Remove( lookupValue );
		} // Remove

		// ----------------------------------------------------------------------
		public void RemoveAt( int index )
		{
			this.values.RemoveAt( index );
		} // RemoveAt

		// ----------------------------------------------------------------------
		public void Clear()
		{
			this.values.Clear();
		} // Clear

		// ----------------------------------------------------------------------
		public ILookupValue[] ToArray()
		{
			ILookupValue[] entries = new ILookupValue[ Count ];
			if ( entries.Length > 0 )
			{
				CopyTo( entries, 0 );
			}
			return entries;
		} // ToArray

		// ----------------------------------------------------------------------
		public void CopyTo( ILookupValue[] array, int index )
		{
			this.values.CopyTo( array, index );
		} // CopyTo

		// ----------------------------------------------------------------------
		public override bool Equals( object obj )
		{
			return CollectionTool.AreEqual( this, obj );
		} // Equals

		// ----------------------------------------------------------------------
		public override int GetHashCode()
		{
			return CollectionTool.ComputeHashCode( this );
		} // GetHashCode

		// ----------------------------------------------------------------------
		public override string ToString()
		{
			return CollectionTool.ToString( this );
		} // ToString

		// ----------------------------------------------------------------------
		// members
		private readonly List<ILookupValue> values = new List<ILookupValue>();
		private readonly string name;

	} // class LookupValueCollection

} // namespace Itenso.WebUserForms.Data.Lookup
// -- EOF -------------------------------------------------------------------
