// -- FILE ------------------------------------------------------------------
// name       : FormXml.cs
// project    : Itenso Web User Forms
// created    : Jani Giannoudis - 2008.10.30
// language   : c#
// environment: .NET 2.0
// copyright  : (c) 2008-2012 by Itenso GmbH, Switzerland
// --------------------------------------------------------------------------
using System.Xml;

namespace Itenso.WebUserForms.Data.Form
{
	using System;

	// ------------------------------------------------------------------------
	public sealed class FormXml : XmlBase<IForm, Form>
	{

		// ----------------------------------------------------------------------
		private FormXml()
			: base( docTag, FormXmlSchema.XmlReadSettings )
		{
		} // FormXml

		// ----------------------------------------------------------------------
		public static FormXml Instance
		{
			get
			{
				if ( instance == null )
				{
					lock ( mutex )
					{
						if ( instance == null )
						{
							instance = new FormXml();
						}
					}
				}
				return instance;
			}
		} // Instance

		// ----------------------------------------------------------------------
		protected override void DoLoad( IForm form, XmlElement formElement )
		{
			form.FormType = GetAttribute( formElement, typeAttr );
			form.FormId = GetAttribute( formElement, idAttr );

			DateTime? dateTimeAttribute = GetDateTimeAttribute( formElement, createdAttr );
			if ( dateTimeAttribute != null )
			{
				form.SetCreated( dateTimeAttribute.Value,
					GetAttribute( formElement, createdByAttr ) );
			}

			if ( formElement.HasAttribute( updatedAttr ) )
			{
				DateTime? timeAttribute = GetDateTimeAttribute( formElement, updatedAttr );
				if ( timeAttribute != null )
				{
					form.MarkUpdated( timeAttribute.Value,
						formElement.HasAttribute( updatedByAttr ) ?
						GetAttribute( formElement, updatedByAttr ) : form.CreatedByUser );
				}
			}
			form.IsLocked = "true".Equals( GetAttribute( formElement, lockedAttr ) );

			LoadFormGroup( form, formElement );
		} // DoLoad

		// ----------------------------------------------------------------------
		private static void LoadFormGroup( IFormGroup formGroup, XmlElement formGroupElement )
		{
			formGroup.Name = GetAttribute( formGroupElement, nameAttr );

			formGroup.Entities.Clear();
			XmlElement formEntityElement = GetFirstChildElement( formGroupElement );
			while ( formEntityElement != null )
			{
				switch ( formEntityElement.LocalName )
				{
					case fieldTag:
						FormField formField = new FormField();
						formField.Name = GetAttribute( formEntityElement, nameAttr );
						formField.Content = formEntityElement.InnerText;
						formGroup.Entities.Add( formField );
						break;
					case formTag:
						FormGroup subFormGroup = new FormGroup();
						LoadFormGroup( subFormGroup, formEntityElement );
						formGroup.Entities.Add( subFormGroup );
						break;
				}
				formEntityElement = GetNextSiblingElement( formEntityElement );
			}
		} // LoadFormGroup

		// ----------------------------------------------------------------------
		protected override void DoSave( IForm form, XmlElement formElement )
		{
			SetAttribute( formElement, typeAttr, form.FormType );
			SetAttribute( formElement, idAttr, form.FormId );
			SetDateTimeAttribute( formElement, createdAttr, form.Created );
			SetAttribute( formElement, createdByAttr, form.CreatedByUser );
			SetDateTimeAttribute( formElement, updatedAttr, form.LastUpdated );
			SetAttribute( formElement, updatedByAttr, form.LastUpdatedByUser );
			SetAttribute( formElement, lockedAttr, form.IsLocked ? "true" : "false" );

			SaveFormGroup( form, formElement );
		} // DoSave

		// ----------------------------------------------------------------------
		private static void SaveFormGroup( IFormGroup formGroup, XmlElement formGroupElement )
		{
			SetAttribute( formGroupElement, nameAttr, formGroup.Name );

			foreach ( IFormEntity formEntity in formGroup.Entities )
			{
				switch ( formEntity.Kind )
				{
					case FormEntityType.Field:
						XmlElement formFieldElement = AppendTag( formGroupElement, fieldTag );
						IFormField formField = formEntity as IFormField;
						if ( formField != null )
						{
							SetAttribute( formFieldElement, nameAttr, formField.Name );
							formFieldElement.InnerText = formField.Content;
						}
						break;
					case FormEntityType.Form:
						XmlElement formSubGroupElement = AppendTag( formGroupElement, formTag );
						IFormGroup formSubGroup = formEntity as IFormGroup;
						SaveFormGroup( formSubGroup, formSubGroupElement );
						break;
				}
			}
		} // SaveFormGroup

		// ----------------------------------------------------------------------
		// members
		private static readonly object mutex = new object();
		private static volatile FormXml instance;

		private const string docTag = "form";
		private const string formTag = "form";
		private const string fieldTag = "field";

		private const string nameAttr = "name";
		private const string typeAttr = "type";
		private const string idAttr = "id";
		private const string createdAttr = "created";
		private const string createdByAttr = "created-by";
		private const string updatedAttr = "updated";
		private const string updatedByAttr = "updated-by";
		private const string lockedAttr = "locked";

	} // class FormXml

} // namespace Itenso.WebUserForms.Data.Form
// -- EOF -------------------------------------------------------------------
