﻿// -- FILE ------------------------------------------------------------------
// name       : UserFormVisitor.cs
// project    : Itenso Web User Forms
// created    : Jani Giannoudis - 2008.10.30
// language   : c#
// environment: .NET 2.0
// copyright  : (c) 2008-2012 by Itenso GmbH, Switzerland
// --------------------------------------------------------------------------
using System;
using System.Web.UI;

namespace Itenso.WebUserForms.Controls
{

	// ------------------------------------------------------------------------
	public abstract class UserFormVisitor
	{

		// ----------------------------------------------------------------------
		protected UserFormVisitor( Control startControl )
		{
			if ( startControl == null )
			{
				throw new ArgumentNullException( "startControl" );
			}

			this.startControl = startControl;
		} // UserFormVisitor

		// ----------------------------------------------------------------------
		public Control StartControl
		{
			get { return this.startControl; }
		} // StartControl

		// ----------------------------------------------------------------------
		protected void Start()
		{
			VisitForm( this.startControl, null );
		} // Start

		// ----------------------------------------------------------------------
		protected virtual void EnterForm( Control control )
		{
		} // EnterForm

		// ----------------------------------------------------------------------
		protected virtual void LeaveForm( Control control )
		{
		} // LeaveForm

		// ----------------------------------------------------------------------
		protected virtual void VisitControl( Control control )
		{
		} // VisitControl

		// ----------------------------------------------------------------------
		protected virtual void VisitHeader( Control control, 
			IUserFormHeader formHeader, IUserFormHeader parentFormHeader )
		{
			if ( parentFormHeader == null )
			{
				VisitMainHeader( control, formHeader );
			}
			else
			{
				VisitSubHeader( control, formHeader, parentFormHeader );
			}
		} // VisitFormHeader

		// ----------------------------------------------------------------------
		protected virtual void VisitMainHeader( Control control, 
			IUserFormHeader formHeader )
		{
		} // VisitMainHeader

		// ----------------------------------------------------------------------
		protected virtual void VisitSubHeader( Control control, 
			IUserFormHeader formHeader, IUserFormHeader parentFormHeader )
		{
		} // VisitSubHeader

		// ----------------------------------------------------------------------
		protected virtual void VisitFormField( Control control,
			IUserFormField formField, IUserFormHeader formHeader )
		{
			IListField listFormField = formField as IListField;
			if ( listFormField != null )
			{
				VisitListFormField( control, listFormField, formHeader );
			}

			ILookupField lookupField = formField as ILookupField;
			if ( lookupField != null )
			{
				VisitLookupFormField( control, lookupField, formHeader );
			}

			IExpressionField expressionField = formField as IExpressionField;
			if ( expressionField != null )
			{
				VisitExpresionFormField( control, expressionField, formHeader );
			}
		} // VisitFormField

		// ----------------------------------------------------------------------
		protected virtual void VisitListFormField( Control control,
			IListField listFormField, IUserFormHeader formHeader )
		{
		} // VisitListFormField

		// ----------------------------------------------------------------------
		protected virtual void VisitLookupFormField( Control control,
			ILookupField lookupField, IUserFormHeader formHeader )
		{
		} // VisitLookupFormField

		// ----------------------------------------------------------------------
		protected virtual void VisitExpresionFormField( Control control,
			IExpressionField expressionField, IUserFormHeader formHeader )
		{
		} // VisitExpresionFormField

		// ----------------------------------------------------------------------
		protected virtual void VisitFormCommand( Control control,
			IUserFormCommand formCommand, IUserFormHeader formHeader )
		{
		} // VisitFormCommand

		// ----------------------------------------------------------------------
		private void VisitForm( Control control, IUserFormHeader parentFormHeader )
		{
			VisitControl( control );
			if ( control.Controls.Count == 0 )
			{
				return;
			}

			IUserFormHeader formHeader = FindFormHeader( control.Controls );
			bool enterForm = false;
			if ( formHeader != null )
			{
				enterForm = true;
				EnterForm( control );
				VisitHeader( control, formHeader, parentFormHeader );
			}
			else
			{
				formHeader = parentFormHeader;
			}

			VisitFormControls( control.Controls, formHeader );

			int controlCount = control.Controls.Count;
			for ( int i = 0; i < controlCount; i++ )
			{
				Control subControl = control.Controls[ i ];
				VisitForm( subControl, formHeader );
				if ( control.Controls.Count != controlCount )
				{
					throw new InvalidOperationException();
				}
			}

			if ( enterForm )
			{
				LeaveForm( control );
			}
		} // VisitForm

		// ----------------------------------------------------------------------
		private static IUserFormHeader FindFormHeader( ControlCollection controls )
		{
			foreach ( Control control in controls )
			{
				IUserFormHeader formHeader = control as IUserFormHeader;
				if ( formHeader != null )
				{
					return formHeader;
				}
			}
			return null;
		} // FindFormHeader

		// ----------------------------------------------------------------------
		private void VisitFormControls( ControlCollection controls, IUserFormHeader formHeader )
		{
			int controlCount = controls.Count;
			for ( int i = 0; i < controlCount; i++ )
			{
				Control control = controls[ i ];

				IUserFormField formField = control as IUserFormField;
				if ( formField != null )
				{
					VisitFormField( control, formField, formHeader );
					if ( controls.Count != controlCount )
					{
						throw new InvalidOperationException();
					}
				}

				IUserFormCommand formCommand = control as IUserFormCommand;
				if ( formCommand != null )
				{
					VisitFormCommand( control, formCommand, formHeader );
				}
			}
		} // VisitFormControls

		// ----------------------------------------------------------------------
		// members
		private readonly Control startControl;

	} // class UserFormVisitor

} // namespace Itenso.WebUserForms.Controls
// -- EOF -------------------------------------------------------------------
