﻿// -- FILE ------------------------------------------------------------------
// name       : UserFormInfoRenderer.cs
// project    : Itenso Web User Forms
// created    : Jani Giannoudis - 2008.10.30
// language   : c#
// environment: .NET 2.0
// copyright  : (c) 2008-2012 by Itenso GmbH, Switzerland
// --------------------------------------------------------------------------
using System;
using System.Text;
using System.Drawing;
using System.Web.UI;
using System.Web.UI.WebControls;

namespace Itenso.WebUserForms.Controls
{
	using System.Globalization;

	// ------------------------------------------------------------------------
	internal class UserFormInfoRenderer : UserFormVisitor
	{

		// ----------------------------------------------------------------------
		public UserFormInfoRenderer( Control startControl )
			: base( startControl )
		{
		} // UserFormInfoRenderer

		// ----------------------------------------------------------------------
		public bool ShowRuntimeHeaderInfo
		{
			get { return this.showRuntimeHeaderInfo; }
			set { this.showRuntimeHeaderInfo = value; }
		} // ShowRuntimeHeaderInfo

		// ----------------------------------------------------------------------
		public bool ShowDesigntimeHeaderInfo
		{
			get { return this.showDesigntimeHeaderInfo; }
			set { this.showDesigntimeHeaderInfo = value; }
		} // ShowDesigntimeHeaderInfo

		// ----------------------------------------------------------------------
		public void Render( Table renderTable )
		{
			if ( renderTable == null )
			{
				throw new ArgumentNullException( "renderTable" );
			}
			this.table = renderTable;
			this.level = 0;

			AddTitleRow( "User Form Details" );
			AddColumnTitleRow( "Name", "Value", "Properties", "Type" );

			Start();
		} // Render

		// ----------------------------------------------------------------------
		protected virtual string GetFormFieldName( IUserFormField formField )
		{
			return formField.FieldName;
		} // GetFormFieldName

		// ----------------------------------------------------------------------
		protected virtual string GetFormFieldValue( IUserFormField formField )
		{
			return formField.FieldValue;
		} // GetFormFieldValue

		// ----------------------------------------------------------------------
		protected virtual string GetFormFieldProperties( IUserFormField formField )
		{
			this.sb.Remove( 0, sb.Length );

			this.sb.Append( formField.AllowEdit ? "Editable" : "Read-Only" );

			IListField listField = formField as IListField;
			if ( listField != null )
			{
				this.sb.Append( ", List[" );
				this.sb.Append( listField.Items.Count.ToString( CultureInfo.InvariantCulture ) );
				this.sb.Append( "]" );
			}

			ILookupField lookupField = formField as ILookupField;
			if ( lookupField != null )
			{
				this.sb.Append( ", Lookup=" );
				this.sb.Append( lookupField.LookupName );
			}

			IExpressionField expressionField = formField as IExpressionField;
			if ( expressionField != null )
			{
				this.sb.Append( ", Expression=" );
				this.sb.Append( expressionField.FieldExpression );
			}

			return this.sb.ToString();
		} // GetFormFieldProperties

		// ----------------------------------------------------------------------
		protected virtual string GetFormFieldControlType( Control control, IUserFormField formField )
		{
			this.sb.Remove( 0, sb.Length );

			this.sb.Append( control.GetType().Name );

			int extendedInfoCount = 0;

			if ( formField is IListField )
			{
				string token = extendedInfoCount == 0 ? " [" : ", ";
				this.sb.Append( token );
				this.sb.Append( "List" );
				extendedInfoCount++;
			}

			if ( formField is ILookupField )
			{
				string token = extendedInfoCount == 0 ? " [" : ", ";
				this.sb.Append( token );
				this.sb.Append( "Lookup" );
				extendedInfoCount++;
			}

			if ( formField is IExpressionField )
			{
				string token = extendedInfoCount == 0 ? " [" : ",";
				this.sb.Append( token );
				this.sb.Append( "Expression" );
				extendedInfoCount++;
			}

			if ( formField is IPlaceholderControl )
			{
				string token = extendedInfoCount == 0 ? " [" : ", ";
				this.sb.Append( token );
				this.sb.Append( "Placeholder" );
				extendedInfoCount++;
			}

			if ( extendedInfoCount > 0 )
			{
				sb.Append( "]" );
			}

			return sb.ToString();
		} // GetFormFieldControlType

		// ----------------------------------------------------------------------
		protected override void EnterForm( Control control )
		{
			this.level++;
		} // EnterForm

		// ----------------------------------------------------------------------
		protected override void LeaveForm( Control control )
		{
			this.level--;
		} // LeaveForm

		// ----------------------------------------------------------------------
		protected override void VisitHeader( Control control, IUserFormHeader formHeader, IUserFormHeader parentFormHeader )
		{
			AddUserControlRow( control as UserControl );

			if ( this.showDesigntimeHeaderInfo )
			{
				AddHeaderRow( "Header.Type", formHeader.Type );
				AddHeaderRow( "Header.Name", formHeader.Name );
				AddHeaderRow( "Header.Description", formHeader.Description );
				AddHeaderRow( "Header.Version", formHeader.Version );
			}

			if ( this.showRuntimeHeaderInfo )
			{
				AddHeaderRow( "Header.FormType", formHeader.FormType );
				AddHeaderRow( "Header.FormName", formHeader.FormName );
				AddHeaderRow( "Header.FormId", formHeader.FormId );
				AddHeaderRow( "Header.FormVersion", formHeader.FormVersion );
				AddHeaderRow( "Header.FormCreatedByUser", formHeader.FormCreatedByUser );
				AddHeaderRow( "Header.FormCreated", formHeader.FormCreated.HasValue ?
					formHeader.FormCreated.Value.ToString( CultureInfo.InvariantCulture ) : string.Empty );
				AddHeaderRow( "Header.FormLastUpdatedByUser", formHeader.FormLastUpdatedByUser );
				AddHeaderRow( "Header.FormLastUpdated", formHeader.FormLastUpdated.HasValue ?
					formHeader.FormLastUpdated.Value.ToString( CultureInfo.InvariantCulture ) : string.Empty );
			}
		} // VisitMainHeader

		// ----------------------------------------------------------------------
		protected override void VisitFormField( Control control, IUserFormField formField, IUserFormHeader formHeader )
		{
			AddFieldRow(
				GetFormFieldName( formField ),
				GetFormFieldValue( formField ),
				GetFormFieldProperties( formField ),
				GetFormFieldControlType( control, formField ) );
		} // VisitFormField

		// ----------------------------------------------------------------------
		private void AddTitleRow( string name )
		{
			TableRow tableRow = AddRow( name, string.Empty, string.Empty, string.Empty );
			tableRow.BackColor = SystemColors.ActiveCaption;
			tableRow.ForeColor = SystemColors.ActiveCaptionText;
			tableRow.Font.Bold = true;
			tableRow.Cells[ 0 ].HorizontalAlign = HorizontalAlign.Center;
			tableRow.Cells[ 0 ].ColumnSpan = tableRow.Cells.Count;
			tableRow.Cells[ 1 ].Visible = false;
			tableRow.Cells[ 2 ].Visible = false;
			tableRow.Cells[ 3 ].Visible = false;
		} // AddTitleRow

		// ----------------------------------------------------------------------
		private void AddColumnTitleRow( string name, string value, string props, string type )
		{
			TableRow tableRow = AddRow( name, value, props, type );
			tableRow.BackColor = SystemColors.ActiveCaption;
			tableRow.ForeColor = SystemColors.ActiveCaptionText;
			tableRow.Font.Bold = true;
			tableRow.Cells[ 0 ].HorizontalAlign = HorizontalAlign.Center;
			tableRow.Cells[ 1 ].HorizontalAlign = HorizontalAlign.Center;
			tableRow.Cells[ 2 ].HorizontalAlign = HorizontalAlign.Center;
			tableRow.Cells[ 3 ].HorizontalAlign = HorizontalAlign.Center;
		} // AddColumnTitleRow

		// ----------------------------------------------------------------------
		private void AddUserControlRow( UserControl userControl )
		{
			TableRow tableRow = AddRow(
				userControl.GetType().Name + " (" + this.level + ")",
				string.Empty,
				string.Empty,
				string.Empty,
				this.level - 1 );
			tableRow.BackColor = SystemColors.Info;
			tableRow.Cells[ 0 ].ColumnSpan = tableRow.Cells.Count;
			tableRow.Cells[ 1 ].Visible = false;
			tableRow.Cells[ 2 ].Visible = false;
			tableRow.Cells[ 3 ].Visible = false;
		} // AddUserControlRow

		// ----------------------------------------------------------------------
		private void AddHeaderRow( string name, string value )
		{
			TableRow tableRow = AddRow( name, value, string.Empty, string.Empty );
			tableRow.BackColor = SystemColors.ControlDark;
		} // AddHeaderRow

		// ----------------------------------------------------------------------
		private void AddFieldRow( string name, string value, string props, string type )
		{
			TableRow tableRow = AddRow( name, value, props, type );
			tableRow.BackColor = SystemColors.Control;
		} // AddFieldRow

		// ----------------------------------------------------------------------
		private TableRow AddRow( string name, string value, string props, string type )
		{
			return AddRow( name, value, props, type, this.level );
		} // AddRow

		// ----------------------------------------------------------------------
		private TableRow AddRow( string name, string value, string props, string type, int rowLevel )
		{
			TableRow tableRow = new TableRow();

			TableCell fieldNameCell = new TableCell();

			this.sb.Remove( 0, sb.Length );
			for ( int i = 0; i < rowLevel; i++ )
			{
				sb.Append( "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;" );
			}
			sb.Append( name );
			fieldNameCell.Text = sb.ToString();
			tableRow.Cells.Add( fieldNameCell );

			TableCell fieldValueCell = new TableCell();
			fieldValueCell.Text = value;
			tableRow.Cells.Add( fieldValueCell );

			TableCell controlPropsCell = new TableCell();
			controlPropsCell.Text = props;
			tableRow.Cells.Add( controlPropsCell );

			TableCell controlTypeCell = new TableCell();
			controlTypeCell.Text = type;
			tableRow.Cells.Add( controlTypeCell );

			this.table.Rows.Add( tableRow );

			return tableRow;
		} // AddRow

		// ----------------------------------------------------------------------
		// members
		private readonly StringBuilder sb = new StringBuilder();
		private Table table;
		private int level;
		private bool showDesigntimeHeaderInfo = true;
		private bool showRuntimeHeaderInfo = true;

	} // class UserFormInfoRenderer

} // namespace Itenso.WebUserForms.Controls
// -- EOF -------------------------------------------------------------------
