// -- FILE ------------------------------------------------------------------
// name       : UserFormFieldDesigner.cs
// project    : Itenso Web User Forms
// created    : Jani Giannoudis - 2008.10.30
// language   : c#
// environment: .NET 2.0
// copyright  : (c) 2008-2012 by Itenso GmbH, Switzerland
// --------------------------------------------------------------------------
using System;
using System.IO;
using System.Text;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.Design;
using System.Drawing;

namespace Itenso.WebUserForms.Controls
{

	// ------------------------------------------------------------------------
	public class UserFormFieldDesigner : ControlDesigner
	{

		// ----------------------------------------------------------------------
		public UserFormFieldDesigner()
		{
		} // UserFormFieldDesigner

		// ----------------------------------------------------------------------
		protected Control DesignControl
		{
			get { return Component as Control; }
		} // DesignControl

		// ----------------------------------------------------------------------
		protected Color DefaultFieldNameColor
		{
			get { return this.defaultFieldNameColor; }
			set { this.defaultFieldNameColor = value; }
		} // DefaultFieldNameColor

		// ----------------------------------------------------------------------
		protected Color MissingFieldNameColor
		{
			get { return this.missingFieldNameColor; }
			set { this.missingFieldNameColor = value; }
		} // MissingFieldNameColor

		// ----------------------------------------------------------------------
		protected Color InvalidFieldNameColor
		{
			get { return this.invalidFieldNameColor; }
			set { this.invalidFieldNameColor = value; }
		} // InvalidFieldNameColor

		// ----------------------------------------------------------------------
		protected string FieldName
		{
			get
			{
				IUserFormField webformField = Component as IUserFormField;
				if ( webformField == null )
				{
					throw new InvalidOperationException( "IUserFormField required" );
				}
				return webformField.FieldName;
			}
		} // FieldName

		// ----------------------------------------------------------------------
		protected string FieldValue
		{
			get
			{
				IUserFormField webformField = Component as IUserFormField;
				if ( webformField == null )
				{
					throw new InvalidOperationException( "IUserFormField required" );
				}
				return webformField.FieldValue;
			}
		} // FieldValue

		// ----------------------------------------------------------------------
		protected bool HasFieldName
		{
			get { return !string.IsNullOrEmpty( FieldName ); }
		} // HasFieldName

		// ----------------------------------------------------------------------
		protected bool HasFieldValue
		{
			get { return !string.IsNullOrEmpty( FieldValue ); }
		} // HasFieldValue

		// ----------------------------------------------------------------------
		protected Control ConflictingControl
		{
			get
			{
				Control designControl = DesignControl;
				if ( designControl == null )
				{
					return null;
				}

				Page page = designControl.Page;
				if ( page == null )
				{
					return null;
				}

				return FindConflictingControl( page.Controls, FieldName );
			}
		} // ConflictingControl

		// ----------------------------------------------------------------------
		protected bool HasConfilictingControl
		{
			get { return ConflictingControl != null; }
		} // HasConfilictingControl

		// ----------------------------------------------------------------------
		public override string GetDesignTimeHtml()
		{
			Control designControl = DesignControl;
			if ( designControl == null )
			{
				return base.GetDesignTimeHtml();
			}

			WebControl designWebControl = designControl as WebControl;
			if ( designWebControl != null )
			{
				designWebControl.BackColor = GetStatusColor();
			}
			return GetRenderHtml( designControl );
		} // GetDesignTimeHtml

		// ----------------------------------------------------------------------
		protected virtual Color GetStatusColor()
		{
			if ( !HasFieldName )
			{
				return MissingFieldNameColor;
			}

			if ( HasConfilictingControl )
			{
				return InvalidFieldNameColor;
			}

			return DefaultFieldNameColor;
		} // GetStatusColor

		// ----------------------------------------------------------------------
		protected virtual string GetRenderHtml( Control control )
		{
			if ( control is WebControl )
			{
				StringWriter text = new StringWriter();
				HtmlTextWriter writer = new HtmlTextWriter( text );
				control.RenderControl( writer );
				return text.ToString();
			}

			StringBuilder sb = new StringBuilder();
			string fieldName = FieldName;
			if ( !string.IsNullOrEmpty( fieldName ) )
			{
				sb.Append( fieldName );
			}
			string fieldValue = FieldValue;
			if ( !string.IsNullOrEmpty( fieldValue ) )
			{
				if ( sb.Length != 0 )
				{
					sb.Append( " - " );
				}
				sb.Append( fieldValue );
			}

			return CreatePlaceHolderDesignTimeHtml( sb.ToString() );
		} // GetRenderHtml

		// ----------------------------------------------------------------------
		private Control FindConflictingControl( ControlCollection controls, string fieldName )
		{
			if ( controls == null || controls.Count == 0 )
			{
				return null;
			}

			foreach ( Control control in controls )
			{
				IUserFormField formField = control as IUserFormField;
				if ( formField != null )
				{
					if ( fieldName.Equals( formField.FieldName ) && control != DesignControl )
					{
						return control;
					}
				}

				if ( control.Controls.Count > 0 )
				{
					Control subControl = FindConflictingControl( control.Controls, fieldName );
					if ( subControl != null )
					{
						return subControl;
					}
				}
			}

			return null;
		} // FindConflictingControl

		// ----------------------------------------------------------------------
		// members
		private Color defaultFieldNameColor = Color.LightGreen;
		private Color missingFieldNameColor = Color.Red;
		private Color invalidFieldNameColor = Color.DarkSalmon;

	} // class UserFormFieldDesigner

} // namespace Itenso.WebUserForms.Controls
// -- EOF -------------------------------------------------------------------
