﻿// -- FILE ------------------------------------------------------------------
// name       : UserFormCodeValidator.cs
// project    : Itenso Web User Forms
// created    : Jani Giannoudis - 2008.10.30
// language   : c#
// environment: .NET 2.0
// copyright  : (c) 2008-2012 by Itenso GmbH, Switzerland
// --------------------------------------------------------------------------
using System;
using System.Reflection;
using System.Collections.Generic;
using System.Web.UI;
using System.Text;

namespace Itenso.WebUserForms.Controls
{

	// ------------------------------------------------------------------------
	public class UserFormCodeValidator
	{

		// ----------------------------------------------------------------------
		public UserFormCodeValidator()
		{
			InitializeExcludedMethods();
			InitializeExcludedFields();
		} // UserFormCodeValidator

		// ----------------------------------------------------------------------
		public List<string> ExcludedMethods
		{
			get { return this.excludedMethods; }
		} // ExcludedMethods

		// ----------------------------------------------------------------------
		public List<string> ExcludedFields
		{
			get { return this.excludedFields; }
		} // ExcludedFields

		// ----------------------------------------------------------------------
		public bool ExcludeControlFields
		{
			get { return this.excludeControlFields; }
			set { this.excludeControlFields = value; }
		} // ExcludeControlFields

		// ----------------------------------------------------------------------
		public virtual void Validate( UserControl userControl )
		{
			ValidateMethods( userControl );
			ValidateFields( userControl );
		} // ValidateForm

		// ----------------------------------------------------------------------
		public void InitializeExcludedMethods()
		{
			this.excludedMethods.Clear();
			this.excludedMethods.Add( "__BuildControl*" );
			this.excludedMethods.Add( "__DataBind*" );
			this.excludedMethods.Add( "get_Profile" );
			this.excludedMethods.Add( "get_SupportAutoEvents" );
			this.excludedMethods.Add( "get_ApplicationInstance" );
			this.excludedMethods.Add( "FrameworkInitialize" );
		} // InitializeExcludedMethods

		// ----------------------------------------------------------------------
		public void InitializeExcludedFields()
		{
			this.excludedFields.Clear();
			this.excludedFields.Add( "__initialized" );
			this.excludedFields.Add( "__stringResource" );
		} // InitializeExcludedFields

		// ----------------------------------------------------------------------
		protected virtual void ValidateMethods( UserControl userControl )
		{
			List<MethodInfo> localMethods = GetLocalMethods( userControl );
			if ( localMethods.Count > 0 )
			{
				StringBuilder sb = new StringBuilder();
				foreach ( MethodInfo localMethod in localMethods )
				{
					if ( sb.Length > 0 )
					{
						sb.Append( ", " );
					}
					sb.Append( localMethod.Name );
				}
				throw new FormSecurityException( "form contains local methods", sb.ToString() );
			}
		} // ValidateMethods

		// ----------------------------------------------------------------------
		protected virtual void ValidateFields( UserControl userControl )
		{
			List<FieldInfo> localFields = GetLocalFields( userControl );
			if ( localFields.Count > 0 )
			{
				StringBuilder sb = new StringBuilder();
				foreach ( FieldInfo localField in localFields )
				{
					if ( sb.Length > 0 )
					{
						sb.Append( ", " );
					}
					sb.Append( localField.Name );
				}
				throw new FormSecurityException( "form contains local fields", sb.ToString() );
			}
		} // ValidateFields

		// ----------------------------------------------------------------------
		private List<MethodInfo> GetLocalMethods( UserControl userControl )
		{
			MethodInfo[] methodInfos = userControl.GetType().GetMethods(
				BindingFlags.Public |
				BindingFlags.NonPublic |
				BindingFlags.Instance |
				BindingFlags.Static |
				BindingFlags.DeclaredOnly
			);

			List<MethodInfo> localMethods = new List<MethodInfo>();
			foreach ( MethodInfo methodInfo in methodInfos )
			{
				if ( IsLocalMethod( methodInfo ) )
				{
					localMethods.Add( methodInfo );
				}
			}

			return localMethods;
		} // GetLocalMethods

		// ----------------------------------------------------------------------
		private List<FieldInfo> GetLocalFields( UserControl userControl )
		{
			FieldInfo[] fieldInfos = userControl.GetType().GetFields(
				BindingFlags.Public |
				BindingFlags.NonPublic |
				BindingFlags.Instance |
				BindingFlags.Static |
				BindingFlags.DeclaredOnly
			);

			List<FieldInfo> localFields = new List<FieldInfo>();
			foreach ( FieldInfo fieldInfo in fieldInfos )
			{
				if ( IsLocalField( fieldInfo ) )
				{
					localFields.Add( fieldInfo );
				}
			}

			return localFields;
		} // GetLocalFields

		// ----------------------------------------------------------------------
		protected virtual bool IsLocalMethod( MethodInfo methodInfo )
		{
			// exclude all known methods
			foreach ( string excludedMethod in this.excludedMethods )
			{
				if ( IsExcludedName( excludedMethod, methodInfo.Name ) )
				{
					return false;
				}
			}
			return true;
		} // IsLocalMethod

		// ----------------------------------------------------------------------
		protected virtual bool IsLocalField( FieldInfo fieldInfo )
		{
			// exclude all control fields
			if ( this.excludeControlFields )
			{
				Type fieldType = fieldInfo.FieldType;
				if ( fieldType.IsSubclassOf( typeof( Control ) ) )
				{
					return false;
				}
			}

			// exclude all known fields
			foreach ( string excludedField in this.excludedFields )
			{
				if ( IsExcludedName( excludedField, fieldInfo.Name ) )
				{
					return false;
				}
			}

			return true;
		} // IsLocalField

		// ----------------------------------------------------------------------
		protected static bool IsExcludedName( string excluded, string name )
		{
			if ( string.IsNullOrEmpty( excluded ) )
			{
				return false;
			}

			if ( excluded.StartsWith( "*" ) )
			{
				if ( name.EndsWith( excluded.Substring( 1 ) ) )
				{
					return true;
				}
			}
			
			if ( excluded.EndsWith( "*" ) )
			{
				if ( name.StartsWith( excluded.Substring( 0, excluded.Length - 1 ) ) )
				{
					return true;
				}
			}

			return name.Equals( excluded );
		} // IsExcludedName

		// ----------------------------------------------------------------------
		// members
		private readonly List<string> excludedMethods = new List<string>();
		private readonly List<string> excludedFields = new List<string>();
		private bool excludeControlFields = true;

	} // class UserFormCodeValidator

} // namespace Itenso.WebUserForms.Controls
// -- EOF -------------------------------------------------------------------
