﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml.Serialization;
using System.Runtime.Serialization;
using System.IO;
using iTextSharp.text;
using iTextSharp.text.html.simpleparser;
using iTextSharp.text.pdf;

namespace CValenzuela.Common
{
    /// <summary>
    /// Creates a PDF Document from HTML and CSS
    /// </summary>
    [Serializable]
    [SoapType(TypeName = "PDFDocument", IncludeInSchema = true, Namespace = "http://www.senseicris.wordpress.com")]
    [XmlRoot(DataType = "PDFDocument", IsNullable = true, Namespace = "http://www.senseicris.wordpress.com")]
    [DataContract]
    public partial class PDFDocument : IPDFDocument
    {
        /// <summary>
        /// Standard internet MIME type for pdf files
        /// </summary>
        public const String MIMETYPE = "application/pdf";

        /// <summary>
        /// Gets or sets the size of the page.
        /// A4 for Letter
        /// </summary>
        /// <value>
        /// The size of the page.
        /// </value>
        [XmlElement]
        [SoapElement]
        [DataMember]
        public iTextSharp.text.Rectangle PageSize { get; set; }
       
        /// <summary>
        /// Gets or sets the left margin.
        /// </summary>
        /// <value>
        /// The left margin.
        /// </value>
        [XmlElement]
        [SoapElement]
        [DataMember]
        public Single LeftMargin { get; set; }
     
        /// <summary>
        /// Gets or sets the right margin.
        /// </summary>
        /// <value>
        /// The right margin.
        /// </value>
        [XmlElement]
        [SoapElement]
        [DataMember]
        public Single RightMargin { get; set; }
     
        /// <summary>
        /// Gets or sets the top margin.
        /// </summary>
        /// <value>
        /// The top margin.
        /// </value>
        [XmlElement]
        [SoapElement]
        [DataMember]
        public Single TopMargin { get; set; }
       
        /// <summary>
        /// Gets or sets the bottom margin.
        /// </summary>
        /// <value>
        /// The bottom margin.
        /// </value>
        [XmlElement]
        [SoapElement]
        [DataMember]
        public Single BottomMargin { get; set; }
       
        /// <summary>
        /// Gets or sets the title.
        /// This will also be the file name
        /// </summary>
        /// <value>
        /// The title.
        /// </value>
        [XmlElement]
        [SoapElement]
        [DataMember]
        public String Title { get; set; }
        
        /// <summary>
        /// Gets or sets the subject.
        /// </summary>
        /// <value>
        /// The subject.
        /// </value>
        [XmlElement]
        [SoapElement]
        [DataMember]
        public String Subject { get; set; }
       
        /// <summary>
        /// Gets or sets the creator.
        /// Person tha created the PDF
        /// </summary>
        /// <value>
        /// The creator.
        /// </value>
        [XmlElement]
        [SoapElement]
        [DataMember]
        public String Creator { get; set; }
        
        /// <summary>
        /// Gets or sets the author.
        /// Person that created the document which was turned into PDF
        /// </summary>
        /// <value>
        /// The author.
        /// </value>
        [XmlElement]
        [SoapElement]
        [DataMember]
        public String Author { get; set; }
        
        /// <summary>
        /// Gets or sets the key words.
        /// Search Keywords
        /// </summary>
        /// <value>
        /// The key words.
        /// </value>
        [XmlElement]
        [SoapElement]
        [DataMember]
        public String KeyWords { get; set; }
        
        /// <summary>
        /// Gets or sets the CSS.
        /// Optional CSS 2.1 Stylesheet for the document
        /// </summary>
        /// <value>
        /// The CSS.
        /// </value>
        [XmlElement]
        [SoapElement]
        [DataMember]
        public String CSS { get; set; }
       
        /// <summary>
        /// Gets or sets the content of the HTML.
        /// </summary>
        /// <value>
        /// The content of the HTML.
        /// </value>
        [XmlElement]
        [SoapElement]
        [DataMember]
        public String HTMLContent { get; set; }
       
        /// <summary>
        /// The PDF in a Byte array
        /// </summary>
        [XmlElement]
        [SoapElement]
        [DataMember]
        public Byte[] PDF
        {
            get
            {
                byte[] bytPDFBytes = null;
                using (MemoryStream ms = new MemoryStream())
                {
                    using (Document document = new Document(PageSize, LeftMargin, RightMargin, TopMargin, BottomMargin))
                    {
                        using (StringReader sr = new StringReader(HTMLContent))
                        {
                            StyleSheet ssCSSStyles = new StyleSheet();
                            List<IElement> lstHTMLObjects = null;

                            ICSSParser styles = new CSSParser(CSS);

                            foreach (var item in styles.Elements)
                            {
                                ssCSSStyles.LoadTagStyle(item.Key, item.Value);
                            }

                            foreach (var item in styles.Classes)
                            {
                                ssCSSStyles.LoadStyle(item.Key, item.Value);
                            }

                            lstHTMLObjects = HTMLWorker.ParseToList(sr, ssCSSStyles);
                            PdfWriter.GetInstance(document, ms);
                            document.Open();
                            document.AddTitle(Title);
                            document.AddSubject(Subject);
                            document.AddAuthor(Author);
                            document.AddCreator(Creator.Default(Author));
                            document.AddCreationDate();
                            document.AddKeywords(KeyWords);


                            document.NewPage();
                            for (int k = 0; k < lstHTMLObjects.Count; k++)
                            {
                                document.Add((IElement)lstHTMLObjects[k]);
                            }
                            document.Close();
                        }
                        bytPDFBytes = ms.ToArray();
                    }
                }
                return bytPDFBytes;
            }
        }
        
        /// <summary>
        /// Initializes a new instance of the <see cref="PDFDocument"/> class.
        /// Defaults to Standard A4 letter size
        /// </summary>
        public PDFDocument()
        {
            PageSize = iTextSharp.text.PageSize.A4;
            LeftMargin = 80;
            RightMargin = 50;
            TopMargin = 30;
            BottomMargin = 65;
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="PDFDocument"/> class.
        /// Defaults to Standard A4 letter size
        /// 
        /// </summary>
        /// <param name="Title">The title.</param>
        /// <param name="Subject">The subject.</param>
        /// <param name="Author">The author.</param>
        /// <param name="Creator">The creator.</param>
        /// <param name="KeyWords">The key words.</param>
        /// <param name="HTMLContent">Content of the HTML.</param>
        /// <param name="CSS">The CSS.</param>
        public PDFDocument(String Title, String Subject, String Author, String Creator, String KeyWords, String HTMLContent, String CSS)
            : this()
        {
            this.Title = Title;
            this.Subject = Subject;
            this.Author = Author;
            this.Creator = Creator;
            this.KeyWords = KeyWords;
            this.HTMLContent = HTMLContent;
            this.CSS = CSS;
        }
    }   
}
