<?php
/* vim: set expandtab tabstop=4 shiftwidth=4: */

/*
    *   Completely separate set_*() things and rendering.
*/
/**
 * Include gradients class
 * @include
 */
require_once './class.imagegradient.php';



/**
 * Effect types
 * @def
 */
define('CL_APPLEEFFECTS_COVERFLOW', 0);
define('CL_APPLEEFFECTS_TOPSITES',  1);
define('CL_APPLEEFFECTS_GAMECARDS', 2);

/**
 * Color theme types
 * @def
 */
define('CL_APPLEEFFECTS_THEMEBLACK', 0);
define('CL_APPLEEFFECTS_THEMEWHITE', 1);



/**
 * Class coverFlow to generate static coverFlow and glass-like reflection
 *
 * PHP 4.3.2 >
 *
 * 2009Aug9 - More 3D
 *
 * @version 0.2.3
 * @author Uku-Kaarel J&otilde;esaar, http://ukj.pri.ee
 * @package coverFlow
 */
class coverflow {


/**
 * Height_of_reflection = image_height / $reflectionHeight.
 *
 * @var flaot $reflectionHeight
 */
var $reflectionHeight = 3.0;

/**
 * GD Image.
 *
 * @var object $image
 * @internal
 */
var $image;

/**
 * Image width.
 *
 * @var int $width
 * @access private
 */
var $width;

/**
 * Image height
 * @var int $height
 * @access private
 */
var $height;

/**
 * Collection of covers.
 * 
 * <code>
 * $this->covers =
 * array( 0 => PHP_GD_Image_object, 
 *        1 => PHP_GD_Image_object, 
 *        2 => PHP_GD_Image_object
 *      );
 * </code>
 *
 * @var array $covers
 * @access private
 */
var $covers = array();

/**
 * Id of centered cover, related with $this->covers
 *
 * @var int $cover1
 */
var $cover1 = -1;


/**
 * Id of last cover.
 *
 * @var int $lastId
 * @access private
 */
var $lastId = -1;

/**
 * General bacground color for coverFlow.
 *
 * @var array
 */
var $bgColor   = array(255, 255, 255);

/**
 * Gradient color for coverFlow, usually light Silver.
 *
 * @var array
 */
var $bgColorGr = array(230, 230, 230);

/**
 * Reflection color for cover.
 *
 * @var array
 */
var $colorRefl = array(230, 230, 230, 40);

/**
 * Reflection shade-to color for cover.
 *
 * @var array
 */
var $colorRefE = array(255, 255, 255, 127);

/**
 * Cache for imagePixelAntialias()
 * @var int $aaX
 * @access private
 */
var $aaX;

/**
 * Cache for imagePixelAntialias()
 * @var int $aaY
 * @access private
 */
var $aaY;

/**
 * Cache for imagePixelAntialias().
 *
 * @var int $aaa
 * @access private
 */
var $aaa = array();




/**
 * Helper array for the HTML imagemap.
 * 
 * <code>
 * $this->htmlImagemap =
 * array( 0 => array(
 *			'title'=>'Image title', 
 *			'url'=>'http://', 
 *			'x'=>0, 
 *			'y'=>0, 
 *			'w'=>1, 
 *			'h'=>1)
 * );
 * </code>
 *
 * Array key is cover id in $this->covers.
 *
 * @var array
 * @access private
 */
var $htmlImagemap = array();



/**
 * HTML imagemap name.
 *
 * @var string $htmlImagemapName
 */
var $htmlImagemapName='coverflow1';


/**
 * Imagamap HTML rendering order.
 *
 * @var array $htmlImagemapOrder
 * @access private
 */
var $htmlImagemapOrder=array();

/**
 * Name of Apple effect
 *
 * @var array $tyAppleEffect
 * @access public
 */
var $tyAppleEffect = 0;
/**
 * Initiate coverflow.
 *
 * @param int $width
 * @param int $height
 * @return bool
 */
function coverflow($width=400, $height=300) {
	$this->image =  imagecreatetruecolor( $width, $height );
	
	imagealphablending($this->image, TRUE);
	if( function_exists('imageantialias')) imageantialias($this->image, TRUE);
	
	$this->width = $width;
	$this->height = $height;
	
	$this->grad = new imagegradient( $this->image );
	
    $this->tyAppleEffect = CL_APPLEEFFECTS_COVERFLOW;
    $this->themeName = -1;
}

/**
 * Set background color to White.
 *
 * @return none
 */
function set_bgWhite() {
	$this->bgColor = imagecolorallocate( $this->image, 255, 255, 255 );
	
	imagefill( $this->image, 0, 0, $this->bgColor);
	
	$this->bgColor = array(255,255,255);
	$this->bgColorGr = array(230, 230, 230);
	$this->colorRefl = array(230, 230, 230, 40);
	$this->colorRefE = array(255, 255, 255, 127);
	$this->themeName = CL_APPLEEFFECTS_THEMEWHITE;
} //set_bgWhite()


/**
 * Set background color to Black.
 *
 * @return void
 */
function set_bgBlack() {
	$this->bgColor = imagecolorallocate( $this->image, 0, 0, 0 );
	
	imagefill( $this->image, 0, 0, $this->bgColor);
	
	$this->bgColor = array(0,0,0);
	$this->bgColorGr = array(25, 25, 25);
	$this->colorRefl = array(25, 25, 25, 40);
	$this->colorRefE = array(0, 0, 0, 127);
	$this->themeName = CL_APPLEEFFECTS_THEMEBLACK;
} // set_bgBlack()


/**
 * Set effect type to coverFlow
 *
 * @return void
 */
function set_use_coverFlow() {
    $this->tyAppleEffect = CL_APPLEEFFECTS_COVERFLOW;
} // set_use_coverFlow()


/**
 * Set effect type to gameCards
 *
 * @return void
 */
function set_use_gameCards() {
    $this->tyAppleEffect = CL_APPLEEFFECTS_GAMECARDS;
} // set_use_gameCards()


/**
 * Set effect type to topSites
 *
 * @return void
 */
function set_use_topSites() {
    $this->tyAppleEffect = CL_APPLEEFFECTS_TOPSITES;
} // set_use_topSites()


/**
 * Get current theme name
 *
 * @return string
 */
function get_themeName() {
    switch($this->themeName) {
        case CL_APPLEEFFECTS_THEMEBLACK: return 'black'; break;
        case CL_APPLEEFFECTS_THEMEWHITE: return 'white'; break;
        default: return 'undefined'; break;
    }
} // get_themeName()


/**
 * Get current effect name
 *
 * @return string
 */
function get_effectName() {
    switch ($this->tyAppleEffect) {
    case CL_APPLEEFFECTS_GAMECARDS: return 'gameCards'; break;
    case CL_APPLEEFFECTS_COVERFLOW: return 'coverFlow'; break;
    case CL_APPLEEFFECTS_TOPSITES:  return 'topSites';  break;
    default: return 'undefined';
    }
} // get_effectName()



/**
 * Set reflection Height.
 *
 * @param int $h
 * @return none
 */
function set_reflectionHeight( $h ) {
	$this->reflectionHeight=$h;
}//set_reflectionHeight()




/**
 * Add page into flipbook.
 *
 * @param string $f filename
 * @param bool $isCover Center this image.
 * @param string $title Optional title for cover
 * @param string $url Optional URL for cover
 * @return int cover id or bool FALSE
 */
function set_cover($f, $title='', $url='') {
	$fe = trim( pathinfo(strtolower($f), PATHINFO_EXTENSION) );
	
	if( is_file( $f ) && is_readable( $f ) ); else return FALSE;
	
	switch( $fe ) {
		case 'jp': case 'jpg':  case 'jpeg':
			$this->covers[] = imagecreatefromjpeg($f);
			break;
		case 'gif':
			$this->covers[] = imagecreatefromgif($f);
			break;
		case 'png':
			$this->covers[] = imagecreatefrompng($f);
			break;
		default:
			$this->covers[] = imagecreatefromstring($f);
	}
	
	$this->lastId += 1;
	
	$this->htmlImagemap[$this->lastId]['url']=$url;
	$this->htmlImagemap[$this->lastId]['title']=$title;
	
	
	return $this->lastId;
}//set_cover()


/**
 * Set cover to centered.
 *
 * @param int $n id of centered cover
 * @return none
 */
function set_coverCentered( $n ) {
	$c = $this->lastId + 1;
	
	if($n>$c) $this->cover1 =  $c ;
	if($n<0) $this->cover1 =  0 ;
	
	$this->cover1 = $n;
}// set_coverCentered()





/**
 * Set title and Url for HTML imagemap.
 *
 * @param int $coverId
 * @param string $title
 * @param string $url
 * @param string $target
 * @param string $class
 * @param string $style
 * @param string $cssid
 * @param string $onMouseOver
 * @param string $onClick
 * @param string $onMouseOut
 * @return none
 */
function set_htmlAtag($coverId, $title='', $url='', $target='',$class='',$style='',$cssid='', $onMouseOver='', $onClick='', $onMouseOut='' ) {
  $this->htmlImagemap[$coverId] = array(
		'title'=>$title, 
		'url'=>$url, 
		'x'=>0, 'y'=>0, 
		'x2'=>0, 'y2'=>0
		);
		
		if( $class != '')$this->htmlImagemap[$coverId]['class'] = $class;
		if( $style != '')$this->htmlImagemap[$coverId]['style'] = $style;
		if( $cssid != '') $this->htmlImagemap[$coverId]['cssid'] = $cssid;
		if( $target != '')$this->htmlImagemap[$coverId]['target'] = $target;
		if( $onMouseOut != '')$this->htmlImagemap[$coverId]['onMouseOut'] = $onMouseOut;
		if( $onMouseOver != '')$this->htmlImagemap[$coverId]['onMouseOver'] = $onMouseOver;
		if( $onClick != '')$this->htmlImagemap[$coverId]['onClick'] = $onClick;
		
}//set_htmlAtag()




/**
 * Set name for HTML imagemap.
 *
 * @param string  $name
 */
function set_htmlImagemapName( $name ) {
	$this->htmlImagemapName = $name;
}//set_htmlImagemapName()


/**
 * Macro: switch_toCompileFn
 *
 * @return void
 */
function switch_toCompileFn() {
    if($this->themeName == -1)
        $this->set_bgWhite();
    
    $this->draw_bg_gradient();

    switch($this->tyAppleEffect) {
        case CL_APPLEEFFECTS_COVERFLOW: $this->compile_coverFlow(); break;
        case CL_APPLEEFFECTS_TOPSITES:  $this->compile_topSites();  break;
        case CL_APPLEEFFECTS_GAMECARDS: $this->compile_gameCards(); break;
    }
} // switch_toCompileFn()



/**
 * Get coverFlow as PNG image.
 *
 * @param string $fn filename is optional
 * @return bool
 */
function get_imagePng( $fn=FALSE ) {

    $this->switch_toCompileFn();
    
	$vc = version_compare(PHP_VERSION, '5.1.2');
	
	if( is_string( $fn ) ) {
	
	
		if(  $vc === 0 ) {
			$r = imagepng($this->image, $fn, 9);
		} elseif ($vc === 1) {
			$r = imagepng($this->image, $fn, 9, PNG_FILTER_PAETH);
		} else {
			$r = imagepng($this->image, $fn);
		}
		
		
	}else {
	
	
		header('Content-type: image/png');
		if(  $vc === 0 ) {
			$r = imagepng($this->image, '', 9);
		} elseif ($vc === 1) {
			$r = imagepng($this->image, '', 9, PNG_FILTER_PAETH);
		} else {
			$r = imagepng($this->image, '');
		}
		
		
	}
	return $r;
		
} // get_imagePng()



/**
 * Get coverFlow as JPEG image.
 *
 * @param string $fn filename is optional
 * @return bool
 */
function get_imageJpeg( $fn = FALSE ) {

    $this->switch_toCompileFn();
	
	if( is_string( $fn ) ){
		
		
		$r = imagejpeg($this->image, $fn, 90);
	
	
	} else {
		
		
		header('Content-type: image/jpeg');
		$r = imagejpeg($this->image, '', 90);
		
		
	}
	return $r;
	
} // get_imageJpeg()





/**
 * Get image map in HTML
 *
 * <code>
 * <MAP name="coverflow1">
 * <AREA href="./covers/koer.png" 
 *       alt="L&otilde;ke (in English: Bonfire)"
 *       title="L&otilde;ke (in English: Bonfire)"
 *       shape="rect" 
 *       coords="105,66,195,134" /></MAP>
 * </code>
 *
 * @return string HTML imagemap
 */
function get_htmlImagemap( ) {

	$html = '<MAP name="' . $this->htmlImagemapName . '">';
	$target = $class = $cssid = $onClick = $onMouseOver = $onMouseOut = '';
	
	foreach($this->htmlImagemapOrder as $coverId) {
		$a = $this->htmlImagemap[$coverId];
		
		if( $a['url'] == '' && $a['title'] == '' ) continue;
		
		if( isset($a['target'])  ) $target = "\n        target=\"" . $a['target'] ."\"\n";
		if( isset($a['class'])  ) $class = "\n        class=\"" . $a['class'] ."\"\n";
		if( isset($a['cssid'])  ) $cssid = "\n        cssid=\"" . $a['cssid'] ."\"\n";
		if( isset($a['onClick'])  ) $onClick = "\n        onClick=\"" . $a['onClick'] ."\"\n";
		if( isset($a['onMouseOver'])  ) $onMouseOver = "\n        onMouseOver=\"" . $a['onMouseOver'] ."\"\n";
		if( isset($a['onMouseOut'])  ) $onMouseOut = "\n        onMouseOut=\"" . $a['onMouseOut'] ."\"\n";
		
	$html .= '
    <AREA href="' . $a['url']. '" 
        alt="' . $a['title']. '"
        title="' . $a['title']. '"'
		. $target . $class . $cssid . $onClick . $onMouseOver . $onMouseOut . '
        shape="rect" 
        coords="' . $a['x']. ',' . $a['y']. ',' .$a['x2']. ',' . $a['y2']. '" />';
	
	}
	$html .= '</MAP>';
	
	return $html;
	
} //get_htmlImagemap()



/**
 * Returns image map name with "#".
 *
 * @return string HTML image map name
 */
function get_htmlImagemapName() {
	return '#' . $this->htmlImagemapName;
} //get_htmlImagemapName()











/**
 * Change image brightness.
 *
 * @param resource $src_im PHP GD image resource
 * @param float $arg1 Brightness level 0.0 - 1.0
 * @return void
 * @access private
 */
function imgFilterBrightness(&$src_im, $arg1=0.0 ) {
		
		//if( function_exists('imagefilter')===TRUE )
		//	return imagefilter( $src_im, IMG_FILTER_BRIGHTNESS,$arg1 );
		
		// if($arg1 < 0.0)$arg1 = 0.0; elseif($arg1 > 1.0) $arg1 = 1.0;
		// +- 0.0 -- 1.0
		$arg1 = intval($arg1 * 255);
		
        $srcsx = imagesx( $src_im );
        $srcsy = imagesy( $src_im );
		
		for ($y = 0; $y < $srcsy; $y++){
			for ($x = 0; $x  <$srcsx; $x++){
				
				$a = imagecolorsforindex($src_im, ImageColorAt($src_im, $x,$y ) );
				
				$a['red']   += $arg1;
				$a['green'] += $arg1;
				$a['blue']  += $arg1;
				
				if($a['red']  <0)$a['red']  =0; elseif($a['red']  >255)$a['red']  =255;
				if($a['green']<0)$a['green']=0; elseif($a['green']>255)$a['green']=255;
				if($a['blue'] <0)$a['blue'] =0; elseif($a['blue'] >255)$a['blue'] =255;
				
				// PHP 4.0.6
				$color = imagecolorresolvealpha($src_im,
					$a['red'],$a['green'],$a['blue'], $a['alpha'] );
				
				imagesetpixel ( $src_im, $x, $y, $color );
				
			}
		}
} //imgFilterBrightness()





/**
 * Compile and set up one cover.
 *
 * @param int $coverId cover id
 * @param int $skew=15 skew angle
 * @param int $dir skew direction
 * @param float $ppw +- 0..1 relative x of center of cover - position on background
 * @param float $pph +- 0..1 relative y of center of cover - position on background
 * @param float $ph 0..1 relative height of cover
 * @param float $shdlvl Shade level, default is 0
 * @return none
 * @access private
 * @todo: build shader() & replace compile_cover() shading capabilies 
 */
function compile_cover($coverId, $skew=15,$dir=0, $ppw=0.2, $pph=0.12, $ph=0.65, $shdlvl=0 ) {
	
	// size of cover source
    $cov_w = imagesx( $this->covers[$coverId] );
    $cov_h = imagesy( $this->covers[$coverId] );
		
	//reflection height && max cover height
	$cov_hm = ($this->height-($this->height/10))*$ph;
	$refl_h   = round ($cov_hm / $this->reflectionHeight);
	$cov_hm  -= $refl_h;
	
	$cov_wm = ($this->width-($this->width/4))*$ph;
	
	list($cov_pw, $cov_ph) =
	 	$this->imageproportion($this->covers[$coverId],$cov_wm,$cov_hm, FALSE);
	
	$imageX =  ($this->width*$ppw)-($cov_pw/2);
	$imageY =  ($this->height*$pph)-($cov_ph/2);
	

	// rf: reflection source is resized, we not waste time for too large images
    $im_r1 = imagecreatetruecolor($cov_pw, $cov_ph);
    imagecopyresampled($im_r1, $this->covers[$coverId], 0, 0, 0, 0, $cov_pw, $cov_ph, $cov_w, $cov_h );
	
    
	// inner shade on rf: 
    switch($this->tyAppleEffect) {
    case CL_APPLEEFFECTS_TOPSITES:
    case CL_APPLEEFFECTS_COVERFLOW:
        if($coverId == $this->cover1) $pageShadeDarkAlpha = 60; else $pageShadeDarkAlpha = 30;
    case CL_APPLEEFFECTS_GAMECARDS:
        $pageShadeDarkAlpha = 65;
        if($skew>1 && count($this->covers)>1 ) {
            $grad1 = new imagegradient( $im_r1 );
            if($dir==0)
            $grad1->left2right(array(72,73,72,127), array(0,0,0,$pageShadeDarkAlpha) ,($cov_pw/2),0, $cov_pw/2, $cov_ph,false);
            else
            $grad1->left2right(array(0,0,0,$pageShadeDarkAlpha),array(72,72,72,127),0,0, $cov_pw/2, $cov_ph,false);
        }
    }
	
    //  ***   !!!   ***
	//farther is darker or lighter, level is stated by $shdlvl
	if( $shdlvl != 0 )
    	$this->imgFilterBrightness($im_r1, $shdlvl );
	
	
	$im_r2 = imagereflect_($im_r1, $refl_h, 'b', $this->colorRefE);
	imagealphablending($im_r2,true);
	
	// lighting on reflection
	$grad = new imagegradient( $im_r2 );
	$grad->top2bottom($this->colorRefE, $this->colorRefl,0,$cov_ph,$cov_pw+1,$refl_h,false);
	$grad->left2right($this->colorRefl, $this->colorRefE,0,$cov_ph,$cov_pw+1,$refl_h,true);
	
	// alphaing the reflaction
	$grad = $this->grad->color_transition($this->colorRefl, $this->colorRefE,$refl_h ,FALSE);
	imagealphablending($im_r2,false);
	imagesavealpha($im_r2,true);
	
	for($y = 0; $y < $refl_h; $y++){
		for($x = 0; $x < $cov_pw; $x++){
			$rgba = imagecolorat($im_r2, $x, $cov_ph + $y);
			$r = 0+(($rgba & 0xFF0000) >> 16);
			$g = 0+(($rgba & 0x00FF00) >> 8);
			$b = 0+($rgba & 0x0000FF);
			imagesetpixel($im_r2, $x, $cov_ph + $y, imagecolorresolvealpha($im_r2, $r, $g, $b, $grad[$y][3]));
		}
	}
	
		
	// skew & placing image with reflection on the background
	$this->imageSkew($this->image, $im_r2, $skew, $dir, $this->colorRefE, $imageX, $imageY);
	
	
	// HTML image map
    $this->htmlImagemap[$coverId]['x']=$imageX;
    $this->htmlImagemap[$coverId]['y']=$imageY;
    $this->htmlImagemap[$coverId]['x2']=$imageX+$cov_pw;
    $this->htmlImagemap[$coverId]['y2']=$imageY+$cov_ph;

	
} // compile_cover()






/**
 * antialias one pixel
 *
 * <code>
 * 0x  x1  xx  xx
 * xx  xx  2x  x3
 * </code>
 *
 * phpversion 4.0.6 >
 *
 * @param resource $im Image
 * @param int $x x pos of pixel to antialias
 * @param int $y y pos of pixel to antialias
 * @param int $n4
 * @author http://ukj.pri.ee
 * @version 1.0.0
 * @access private
 * @return none
 */
function imagePixelAntialias(&$im, $x, $y, $n4=0) {
		
	if( $this->aaX != $x || $this->aaY != $y) {
		
		$this->aaX = $x;
		$this->aaY = $y;
		
		if( $n4 == 0 ) {
		$a0 = imagecolorsforindex($im, ImageColorAt($im, $x,$y ) ); //
		$a1 = imagecolorsforindex($im, ImageColorAt($im,          $x+1,$y ) );
		$a2 = imagecolorsforindex($im, ImageColorAt($im, $x,$y+1 ) );
		$a3 = imagecolorsforindex($im, ImageColorAt($im,          $x+1,$y+1 ) );
		}
		elseif( $n4 == 1 ) {
		$a0 = imagecolorsforindex($im, ImageColorAt($im,          $x,$y ) ); //
		$a1 = imagecolorsforindex($im, ImageColorAt($im, $x-1,$y ) );
		$a2 = imagecolorsforindex($im, ImageColorAt($im, $x-1,$y+1 ) );
		$a3 = imagecolorsforindex($im, ImageColorAt($im,          $x,$y+1 ) );
		}
		elseif( $n4 == 2 ) {
		$a0 = imagecolorsforindex($im, ImageColorAt($im, $x,$y ) ); //
		$a1 = imagecolorsforindex($im, ImageColorAt($im, $x,$y-1 ) );
		$a2 = imagecolorsforindex($im, ImageColorAt($im,          $x+1,$y-1 ) );
		$a3 = imagecolorsforindex($im, ImageColorAt($im,          $x+1,$y ) );
		}
		elseif( $n4 == 3 ) {
		$a0 = imagecolorsforindex($im, ImageColorAt($im,          $x,$y ) ); //
		$a1 = imagecolorsforindex($im, ImageColorAt($im, $x-1,$y-1 ) );
		$a2 = imagecolorsforindex($im, ImageColorAt($im,          $x,$y-1 ) );
		$a3 = imagecolorsforindex($im, ImageColorAt($im, $x-1,$y ) );
		}
		
		$this->aaa = array($a0, $a1, $a2, $a3);
	
	} else {
		$a0=$this->aaa[0]; $a1=$this->aaa[1]; $a2=$this->aaa[2]; $a3=$this->aaa[3];
	}
	
	if( $n4 == 0 ) {
			$a0['red'] = ( ($a0['red']*1.2) + $a1['red'] + $a2['red'] + ($a3['red']/1.2) ) / 4;
		$a0['green'] = ( ($a0['green']*1.2) + $a1['green'] + $a2['green'] + ($a3['green']/1.2) ) / 4;
		$a0['blue'] = ( ($a0['blue']*1.2) + $a1['blue'] + $a2['blue'] + ($a3['blue']/1.2) ) / 4;
		$a0['alpha'] = ( ($a0['alpha']*1.2) + $a1['alpha'] + $a2['alpha'] + ($a3['alpha']/1.2) ) / 4;
	
	
	} elseif( $n4 == 1 ) {
		$a0['red'] = ( ($a0['red']*1.2) + $a1['red'] + ($a2['red']/1.2) + $a3['red'] ) / 4;
		$a0['green'] = ( ($a0['green']*1.2) + $a1['green'] + ($a2['green']/1.2) + $a3['green'] ) / 4;
		$a0['blue'] = ( ($a0['blue']*1.2) + $a1['blue'] + ($a2['blue']/1.2) + $a3['blue'] ) / 4;
		$a0['alpha'] = ( ($a0['alpha']*1.2) + $a1['alpha'] + ($a2['alpha']/1.2) + $a3['alpha'] ) / 4;
	
	
	} elseif( $n4 == 2 ) {
		$a0['red'] = ( ($a0['red']*1.2) + $a1['red'] + ($a2['red']/1.2) + $a3['red'] ) / 4;
		$a0['green'] = ( ($a0['green']*1.2) + $a1['green'] + ($a2['green']/1.2) + $a3['green'] ) / 4;
		$a0['blue'] = ( ($a0['blue']*1.2) + $a1['blue'] + ($a2['blue']/1.2) + $a3['blue'] ) / 4;
		$a0['alpha'] = ( ($a0['alpha']*1.2) + $a1['alpha'] + ($a2['alpha']/1.2) + $a3['alpha'] ) / 4;
	
	
	} elseif( $n4 == 3 ) {
		$a0['red'] = ( ($a0['red']*1.2) + ($a1['red']/1.2) + $a2['red'] + $a3['red'] ) / 4;
		$a0['green'] = ( ($a0['green']*1.2) + ($a1['green']/1.2) + $a2['green'] + $a3['green'] ) / 4;
		$a0['blue'] = ( ($a0['blue']*1.2) + ($a1['blue']/1.2) + $a2['blue'] + $a3['blue'] ) / 4;
		$a0['alpha'] = ( ($a0['alpha']*1.2) + ($a1['alpha']/1.2) + $a2['alpha'] + $a3['alpha'] ) / 4;
	}
	
	if( $a0['red'] > 255 ) $a0['red'] = 255; if( $a0['red'] < 0 ) $a0['red'] = 0;
	if( $a0['green'] > 255 ) $a0['green'] = 255; if( $a0['green'] < 0 ) $a0['green'] = 0;
	if( $a0['blue'] > 255 ) $a0['blue'] = 255; if( $a0['blue'] < 0 ) $a0['blue'] = 0;
	if( $a0['alpha'] > 127 ) $a0['alpha'] = 127; if( $a0['alpha'] < 0 ) $a0['alpha'] = 0;
	
	imagesetpixel($im, $x, $y, imagecolorresolvealpha($im, $a0['red'],$a0['green'],$a0['blue'],$a0['alpha'] ) );
	
} //imagePixelAntialias()






/**
 * Skews an image handle by some angle either left or right.
 *
 * First implementation of this function from Peter Higgins, (Dojo, JS toolkit)
 *
 * @author Uku-Kaarel J&otilde;esaar, http://ukj.pri.ee
 * @version 2.0.0
 * @param resource $im_dst
 * @param resource $im_src 
 * @param int $angle Optional, default is 15
 * @param int $dir Optional, default is 0 
 * @param array $bgColor Optional, default is array(255,255,255,127)
 * @param int $ix Optional, default is 0 
 * @param int $iy Optional, default is 0 
 * @return bool
 * @access private
 */
function imageSkew(&$im_dst, &$im_src, $angle=15, $dir = 0, $bgColor=array(255,255,255,127), $ix=0, $iy=0) {
	
	if( !is_resource( $im_dst ) && !is_resource( $im_src ) ) return FALSE;
	
	$w = imagesx($im_src);
	$h = imagesy($im_src);
	
	// Pixel differences 
	$diff = ($angle / 90); 
	$diff2 = $diff*2;
	// Loop trough each width pixel 
	$currentHeight = $h; 
	$currentY = 0;
	if($dir == 1){
		$currentHeight = 0; 
		$currentY = $h;
	}
	
	if( $dir == 0 ) {
		for($i = 0; $i < $w; $i++) {
			imagecopyresampled($im_dst, $im_src, $ix+$i, $iy+$currentY, $i, 0, 1, $currentHeight, 1, $h);
	
			if($angle > 0 ) {
 				$this->imagePixelAntialias($im_dst, $ix+$i, $iy+$currentY, 0);
 				$this->imagePixelAntialias($im_dst, $ix+$i, $iy+$currentY, 3);
 				$this->imagePixelAntialias($im_dst, $ix+$i, $iy+$currentY, 2);
 				$this->imagePixelAntialias($im_dst, $ix+$i-2, $iy+$currentY-1, 2);
			}	
			
			$currentHeight = $currentHeight - $diff2; 
			$currentY = ($h - $currentHeight) / 2;
		}
		
		
	}else{
		for($i = 0; $i < $w; $i++) {
			imagecopyresampled($im_dst, $im_src, $ix+($w - $i), $iy+$currentY, ($w - $i), 0, 1, $currentHeight, 1, $h); 
			
			if($angle > 0 ) {
				$this->imagePixelAntialias($im_dst, $ix+($w-$i)+$diff2+1, $iy+$currentY-1, 0);
				$this->imagePixelAntialias($im_dst, $ix+($w-$i)+$diff2+1, $iy+$currentY-1, 1);
				$this->imagePixelAntialias($im_dst, $ix+($w-$i)+$diff2+1, $iy+$currentY-2, 2);
				$this->imagePixelAntialias($im_dst, $ix+($w-$i)+$diff2+1, $iy+$currentY-2, 3);
			}
			
			$currentHeight = $h - ( $i * $diff2 );
			$currentY = ($h - $currentHeight) / 2; 
		}
		
	}
	
	return TRUE;
}//imageSkew()




/**
 * Compile coverflow.
 *
 * @return none
 * @access private
 */
function compile_coverFlow() {
	/* $this->covers[]; $this->cover; */
	$c = $this->lastId + 1;
	
	if($this->cover1 < 0 || !isset($this->cover1) ) 
		$ic = $c-1;
	else 
		$ic = $this->cover1;
	
	$i0 =  $i1 = $i2 = 0;
	
	if( $this->themeName == 1 ) {
	    $shd4 = 0.50; $shd3 = 0.40; $shd2 = 0.20; $shd1 = 0.1;
    }elseif( $this->themeName == 0 ) {
	    $shd4 = -0.50; $shd3 = -0.40; $shd2 = -0.20; $shd1 = -0.1;
	}
    
    
    
    
	if( $c == 1 ) {
	
		$this->compile_cover(0, 10,0, 0.5, 0.45, 0.90 );
		$this->htmlImagemapOrder[] = 0;
		
	}
    
    
	elseif( $c == 2 ) {
		if($ic > 0 ) $i0 = 0; else $i0 = 1;
		
		//right
		$this->compile_cover($i0, 	10,1, 0.7, 0.45, 0.55 );
		$this->htmlImagemapOrder[] = $i0;
		
		//left
		$this->compile_cover($ic, 8,0, 0.35, 0.45, 0.75 );
		$this->htmlImagemapOrder[] = $ic;

        $this->htmlImagemap[$i0]['x'] = $this->htmlImagemap[$ic]['x2'] + 1;

	}
    
    
    
	elseif( $c == 3 ) {
		// on the reel
		if($ic==0){$i0=2;$i1=1;} if($ic==1){$i0=2;$i1=0;} if($ic==2){$i0=1;$i1=0;}
		//left
		$this->compile_cover( $i0, 10,0, 0.25, 0.5, 0.50 );
		$this->htmlImagemapOrder[] = $i0;
		
		//right
		$this->compile_cover( $i1, 10,1, 0.75, 0.5, 0.50 );
		$this->htmlImagemapOrder[] = $i1;
		
		//center
		$this->compile_cover( $ic, 0,0, 0.5, 0.5, 0.7 );
		$this->htmlImagemapOrder[] = $ic;
		
        $this->htmlImagemap[$i0]['x2'] = $this->htmlImagemap[$ic]['x'] - 1;
        $this->htmlImagemap[$i1]['x'] = $this->htmlImagemap[$ic]['x2'] + 1;
        
		
		
	} elseif( $c < 1 ) {
	} else {
		
		//left
		if( isset( $this->covers[$ic-4] ) ) {
			$this->compile_cover( $ic-4, 16,0, 0.1, 0.38, 0.19, $shd4 );
			$this->htmlImagemapOrder[] = $ic-4;
        }
		if( isset( $this->covers[$ic-3] ) ) {
			$this->compile_cover( $ic-3, 14,0, 0.2, 0.42, 0.24, $shd3 );
			$this->htmlImagemapOrder[] = $ic-3;

            if( isset( $this->covers[$ic-4] ) ) {
                $this->htmlImagemap[$ic-4]['x2'] = $this->htmlImagemap[$ic-3]['x'] - 1;
            }
        }
		if( isset( $this->covers[$ic-2] ) ) {
			$this->compile_cover( $ic-2, 12,0, 0.3, 0.45, 0.27, $shd2 );
			$this->htmlImagemapOrder[] = $ic-2;

            if( isset( $this->covers[$ic-3] ) ) {
                $this->htmlImagemap[$ic-3]['x2'] = $this->htmlImagemap[$ic-2]['x'] - 1;
            }
        }
		if( isset( $this->covers[$ic-1] ) ) {
			$this->compile_cover( $ic-1, 10,0, 0.4, 0.50, 0.30, $shd1 );
			$this->htmlImagemapOrder[] = $ic-1;
            if( isset( $this->covers[$ic-2] ) ) {
                $this->htmlImagemap[$ic-2]['x2'] = $this->htmlImagemap[$ic-1]['x'] - 1;
            }
        }
        
        
		//right
		if( isset( $this->covers[$ic+4] ) ) {
			$this->compile_cover( $ic+4, 16,1, 0.9, 0.38, 0.19, $shd4 );
			$this->htmlImagemapOrder[] = $ic+4;
        }
		if( isset( $this->covers[$ic+3] ) ) {
			$this->compile_cover( $ic+3, 14,1, 0.8, 0.42, 0.24, $shd3 );
			$this->htmlImagemapOrder[] = $ic+3;
            if( isset( $this->covers[$ic+4] ) ) {
                $this->htmlImagemap[$ic+4]['x'] = $this->htmlImagemap[$ic+3]['x2'] + 1;
            }
        }
		if( isset( $this->covers[$ic+2] ) ) {
			$this->compile_cover( $ic+2, 12,1, 0.7, 0.45, 0.27, $shd2 );
			$this->htmlImagemapOrder[] = $ic+2;
            if( isset( $this->covers[$ic+3] ) ) {
                $this->htmlImagemap[$ic+3]['x'] = $this->htmlImagemap[$ic+2]['x2'] + 1;
            }
        }
		if( isset( $this->covers[$ic+1] ) ) {
			$this->compile_cover( $ic+1, 10,1, 0.6, 0.50, 0.30, $shd1 );
			$this->htmlImagemapOrder[] = $ic+1;
            if( isset( $this->covers[$ic+2] ) ) {
                $this->htmlImagemap[$ic+2]['x'] = $this->htmlImagemap[$ic+1]['x2'] + 1;
            }
        }

		
		//center
		$this->compile_cover( $ic, 0,0, 0.5, 0.5, 0.4 );
		$this->htmlImagemapOrder[] = $ic;
        
        //left -1
        if( isset( $this->covers[$ic-1] ) ) {
            $this->htmlImagemap[$ic-1]['x2'] = $this->htmlImagemap[$ic]['x'] - 1;
        }
        //right +1
        if( isset( $this->covers[$ic+1] ) ) {
            $this->htmlImagemap[$ic+1]['x'] = $this->htmlImagemap[$ic]['x2'] + 1;
        }
		
	}
} //compile_coverFlow()


/**
 * Compile gameCards.
 *
 * @return none
 * @access private
 */
function compile_gameCards() {
    $c = $this->lastId + 1;
    
    //right
    if(isset($this->covers[3])) {
        $this->compile_cover(3, 8, 1, 0.8, 0.2, 0.35 );
        $this->htmlImagemapOrder[] = 3;
    }
    
    //right
    if(isset($this->covers[2])) {
        $this->compile_cover(2, 3 ,1, 0.64, 0.3, 0.4 );
        $this->htmlImagemapOrder[] = 2;
    }
    
    //left
    if(isset($this->covers[1])) {
        $this->compile_cover(1, 5,0, 0.44, 0.35, 0.55 );
        $this->htmlImagemapOrder[] = 1;
    }
    
    //left
    if(isset($this->covers[0])) {
        $this->compile_cover(0, 9,0, 0.25, 0.45, 0.6 );
        $this->htmlImagemapOrder[] = 0;
    }

    $this->htmlImagemap[0]['x'] = $this->htmlImagemap[0]['x2'] + 1;
    
} //compile_gameCards()



function compile_topSites() {
    
    
    
} //compile_topSites()




/**
 * Draw gradient.
 *
 * @access private
 * @return none
 */
function draw_bg_gradient() {

	$gw = $this->width / 15;
	$cz = $this->height / 15;
	$gy = $cz * 3;
	
	if( $this->tyAppleEffect == CL_APPLEEFFECTS_COVERFLOW ) {
        $this->grad->top2bottom(
            $this->bgColorGr, $this->bgColor,
            $gy,$gy*2, 
            ($this->width - ($gy*2)) ,$gy,
            false );
        
        $this->grad->gradient_cor(
            $this->bgColorGr,$this->bgColor,
            0,$gy*2,
            $gy,'bl' );//left
            
        $this->grad->gradient_cor(
            $this->bgColorGr,$this->bgColor,
            $this->width - $gy, $gy*2,
            $gy,'br' );//right
    }
    
    
    elseif( $this->tyAppleEffect == CL_APPLEEFFECTS_GAMECARDS ) {
        $this->grad->top2bottom(
            $this->bgColorGr, $this->bgColor, 0, 0, $this->width, $this->height, true );
    }
    elseif( $this->tyAppleEffect == CL_APPLEEFFECTS_TOPSITES ) {
	    // there is no gradient
    }
    
} //draw_bg_gradient()




/**
 * Calculate new proportional size for image.
 *
 * @param resource $im GD Image
 * @param int $newwidth -1  $newwidth = image width
 * @param int $newheight -1 $newheight =  image height
 * @param bool $tag if TRUE, return string IMG image dimension atributes
 * @return mixed array(width, height, quotient_width<>height )
 * @access private
 */
function imageproportion(&$im,$newwidth=-1,$newheight=-1,$tag=FALSE) {
	
	if(is_resource($im)) {
	$W = imagesx($im);
	$H = imagesy($im);
	} else {
	/*1 = GIF, 2 = JPG, 3 = PNG*/
	list($W,$H,$type,$attr)=getimagesize($im);
	}
	
	$newwidth = (int) $newwidth;
	$newheight = (int) $newheight;
	
	if($newwidth>$W)$newwidth=$W;
	
	if($newheight>$W)$newheight=$W;
	
	// 
	if( $newwidth  <1 and $newheight >0) {$newwidth = $newheight; }
	if( $newheight <1 and $newwidth  >0) {$newheight = $newwidth; }
	
	// do nothing
	if( $newwidth  <1 and $newheight <1) {$newwidth = $W; $newheight = $H;}
	
	/*proportion*/
	//if image side is longer than allowed maximum lenght
	if($W>$H){
		if($W>$newwidth){
			$tW=$newwidth;
			$tH=ceil(($H*$newwidth)/$W);
		}else{
			// no changes if new width is longer than original
			$tW=$W;
			$tH=$H;
		}
	}elseif($W<$H){
		if($H>$newheight){
			$tW=ceil(($W*$newheight)/$H);
			$tH=$newheight;
		}else{
			// no changes if new height is longer than original
			$tW=$W;
			$tH=$H;
		}
	}else{
		if($W>$newwidth)
			$tW=$tH=$newwidth;
		else
			$tW=$tH=$W;
	}
	

	if($tag===TRUE)
		return " width=\"$tW\" height=\"$tH\" ";
	
	$s=1;
	if($W>$H) $s=$W/$H; else $s=$H/$W;
	
	return array( (int)$tW, (int)$tH, $s);

}//imageproportion()



	
}//class coverflow{}









/**
 * General function to generate image with reflection , without shading & opacity.
 *
 * Reflection gave got to be top || bottom || right || left
 *
 * Usage:
 * <code>
 * $res_im = imagereflect_($resource_im, 70, 't');
 * </code>
 *
 * @param resource $im source image
 * @param int $n reflection width if ($side==top||bottom) and height if ($side==left||right).
 * @param string $side t b l r Reflection goes to [top bottom left right] of the image.
 * @param array $bgColor default is array(255,255,255,127)
 * @return resource, destination image.
 */
function imagereflect_( &$im, $n, $side='b', $bgColor=array(255,255,255,127) ) {
    
    if( isset($bgColor[3]) ) {
    	if( $bgColor[3] > 127 ) $bgColor[3] = 127;
    	elseif( $bgColor[3] < 0 ) $bgColor[3] = 0;
	}
    
    $im_w = imagesx($im);
    $im_h = imagesy($im);
    
    if($side == 'l' || $side == 'r' ) {
        $im_r_w = $im_w + $n;
        $im_r_h = $im_h;
        }
    if($side == 't' || $side == 'b' ) {
        $im_r_h = $im_h + $n;
        $im_r_w = $im_w;
        }
            
    $im_r = imagecreatetruecolor($im_r_w, $im_r_h);
    imagealphablending($im_r, FALSE);
	imagesavealpha($im_r, TRUE);
	
    //To Do: if no alphlending or somthing bad, then blend herem get pixel, set pixel
	if( function_exists( 'imagecolorallocatealpha') )
		$bgColor_a = imagecolorallocatealpha($im_r, $bgColor[0],$bgColor[1],$bgColor[2], 127);
	else
    	$bgColor_a = imagecolorallocate($im_r,$bgColor[0],$bgColor[1],$bgColor[2] );
		
    imagefill($im_r, 0, 0, $bgColor_a);
    
        if($side == 'b') { 
        	imagecopy($im_r, $im, 0, 0, 0, 0, $im_w, $im_h); 
        	
			//	Rotate and flip it (strip flip method)
			for ($y = 0; $y < $n; $y++) {
			   imagecopy($im_r, $im, 0, $im_h + $y, 0, $im_h - $y - 1, $im_r_w, 1);
			}
        }
    elseif($side == 't') { 
    	imagecopy($im_r, $im, 0, $n, 0, 0, 	$im_w, $im_h); 

		for ($y = 0; $y < $n; $y++) {
		   imagecopy($im_r, $im, 0, $n - $y - 1, 0, $y + 1, $im_r_w, 1);
		}
    }
    elseif($side == 'l') { 
    	imagecopy($im_r, $im, $n, 0, 0, 0, $im_w, $im_h); 

		for ($x = 0; $x < $n; $x++) {
		   imagecopy($im_r, $im, $n-$x-1, 0, $x+1, 0, 1, $im_h);
		}    	
    }
    elseif($side == 'r') { 
    	imagecopy($im_r, $im, 0, 0, 0, 0, $im_w, $im_h); 
    	
		for ($x = 0; $x < $n; $x++) {
		   imagecopy($im_r, $im, $im_w+$x, 0, $im_w-$x-1, 0, 1, $im_h);
		}    
    }
    

    
    
    return $im_r;
} // imagereflect_()



?>