﻿' Author: Razi Syed.
' www.RaziSyed.com (c) 2012
'
' Distributed under GPL License
' Please keep this Credit/Reference intact

Imports System.Web
Imports System.ComponentModel
Imports System.Web.UI.WebControls

Public Class ObjectBinder(Of T As {Class, New, IEquatable(Of T)})

#Region "Properties"

    ''' <summary>
    ''' Constant Identifier used for Session so that values are retained between Postbacks
    ''' </summary>
    Private ReadOnly Property ID As String
        Get
            Return GetType(T).ToString.Replace(".", "_")
        End Get
    End Property

    Private Property Binder As List(Of T)
        Get
            If HttpContext.Current.Session(ID) Is Nothing Then
                HttpContext.Current.Session(ID) = New List(Of T)
            End If

            Return DirectCast(HttpContext.Current.Session(ID), List(Of T))

        End Get
        Set(value As List(Of T))
            HttpContext.Current.Session(ID) = value
        End Set
    End Property

#End Region

#Region "Constructors"

    Sub New()
        ' Empty Constructor
    End Sub

    Public Sub New(instance As T)
        SetInstance(instance)
    End Sub

    Public Sub New(list As List(Of T))
        SetList(list)
    End Sub

    Sub New(collection As ICollection(Of T))
        SetList(collection)
    End Sub

#End Region

#Region "Methods"

    ''' <summary>
    ''' Get the ObjectDataSource to bind to editable ASP .Net controls like Grid, Formview, etc.
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Function GetDataSource() As ObjectDataSource
        Dim ods = New ObjectDataSource
        With ods
            .ID = "ods" & ID
            .DataObjectTypeName = GetType(T).ToString
            .TypeName = Me.GetType.ToString
            .OldValuesParameterFormatString = "original_{0}"
            .SelectMethod = "List"
            .InsertMethod = "Insert"
            .UpdateMethod = "Update"
            .DeleteMethod = "Delete"

        End With

        Return ods
    End Function

    ''' <summary>
    ''' Gets a single instance if not using a list
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks>Use this to get object when binding to FormView or other controls that do not work with a list</remarks>
    Public Function Instance() As T
        If Binder.Count = 1 Then
            Return Binder(0)
        Else
            Return Nothing
        End If
    End Function

    ''' <summary>
    ''' Manually Dispose the list so that List saved in session is also cleared up
    ''' </summary>
    ''' <remarks></remarks>
    Public Sub Clear()
        Binder.Clear()
        Binder = Nothing
    End Sub

    <DataObjectMethod(DataObjectMethodType.Select, True)> _
    Public Function List() As List(Of T)
        Return Binder
    End Function

    <DataObjectMethod(DataObjectMethodType.Insert, True)> _
    Public Sub Insert(ByVal Instance As T)
        Binder.Add(Instance)
    End Sub

    <DataObjectMethod(DataObjectMethodType.Update, True)> _
    Public Sub Update(ByVal Instance As T)
        If Binder.Count = 1 Then
            ' Update single instance
            Binder(0) = Instance
        Else
            ' Update item in list
            Binder.Remove(Instance) ' Works b/c T must implement IEquatable
            Binder.Add(Instance)
        End If
    End Sub

    <DataObjectMethod(DataObjectMethodType.Delete, True)> _
    Public Sub Delete(ByVal Instance As T)
        Binder.Remove(Instance)
    End Sub

    Public Sub SetList(list As List(Of T))
        Binder = list
    End Sub

    Public Sub SetList(collection As ICollection(Of T))
        Binder = CType(collection, List(Of T))
    End Sub

    Public Sub SetInstance(instance As T)
        Clear()
        Insert(instance)
    End Sub

#End Region

End Class