<?php

  /**------------------------------------------------------------------------------
   * Title:        buffer
   * Filename:     buffer.class.php
   * Version:      1.0
   * Author:       Richard Keizer
   * Email:        ra dot keizer at gmail dot com
   *-------------------------------------------------------------------------------
   * COPYRIGHT (c) 2011 Richard Keizer
   *
   * The source code included in this package is free software; you can
   * redistribute it and/or modify it under the terms of the GNU General Public
   * License as published by the Free Software Foundation. This license can be
   * read at:
   *
   * http://www.opensource.org/licenses/gpl-license.php
   *
   * This program is distributed in the hope that it will be useful, but WITHOUT
   * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
   * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
   *------------------------------------------------------------------------------
   *
   *
   * Implementation of a buffer that is able to receive binary data and track
   * upload speed and progress.
   * Provides user callbacks for notification
   */
  
  class Buffer {
    protected $data = '';
    protected $datasize = 0;
    protected $maxdatasize = null;      //optional, used for 'estimated time' and 'percentage done'
    
    protected $transferspeed = 0;       //bytes per second
    protected $prevdatasize = 0;
    protected $prevtime;
    
    protected $progress = null;
    protected $prevprogress = null;
    
    protected $onSpeedChange = null;
    protected $onProgressChange = null;
    
    public function __construct($onspeedchange=null, $onprogresschange=null) {
      $this->onSpeedChange = $onspeedchange;
      $this->onProgressChange = $onprogresschange;
      $this->init();
    }
    
    public function init() {
      $this->data = '';
      $this->datasize = 0;
      $this->maxdatasize = null;
      $this->transferspeed = 0;
      $this->prevdatasize = 0;
      
      $this->progress = null;
      $this->prevprogress = null;
      
      $this->prevtime = microtime(true);
    }
    
    public function getData() {
      return $this->data;
    }
    
    public function getSize() {
      return $this->datasize;
    }
    
    public function getSpeed() {
      return round($this->transferspeed/1000);
    }
    
    public function getProgress() {
      return $this->progress;
    }
    
    public function setMaxSize($size) {
      $this->maxdatasize = $size;
    }
    
    public function getMaxSize() {
      return $this->maxdatasize;
    }
    
    public function append($data) {
      $this->data .= $data;
      $this->datasize += strlen($data);
      
      $this->updateProgress();
      $this->updateSpeed();
    }
    
    public function hasLimitedSize() {
      return !is_null($this->maxdatasize);
    }
    
    public function isFull() {
      return $this->hasLimitedSize() && ($this->getSize() >= $this->getMaxSize());
    }
    
    protected function updateProgress() {
      if (isset($this->maxdatasize)) {
        $this->progress = round(  ($this->getSize() /$this->maxdatasize)*100);
        if ($this->progress != $this->prevprogress) {
          $this->prevprogress = $this->progress;
          if (is_callable($this->onProgressChange))
            call_user_func_array($this->onProgressChange, array());
        }
      }
    }
    
    protected function updateSpeed() {
      $now = microtime(true);
      $lapse = $now - $this->prevtime;
      if ($lapse >= 1) {
        $this->transferspeed = round(($this->datasize - $this->prevdatasize) / $lapse);
        $this->prevdatasize = $this->datasize;
        $this->prevtime = $now;
        if (is_callable($this->onSpeedChange))
          call_user_func_array($this->onSpeedChange, array());
      }      
    }
  }

