using System;
using System.Web.UI;
using System.Web.UI.Design;
using System.Web.UI.WebControls;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Design;
using System.ComponentModel.Design;

namespace LoadingControl
{
    /// <summary>
    /// A custom control that shows loading message while Page loads
    /// </summary>
    [DefaultProperty("LoadingText")]
    [ToolboxData("<{0}:Loading runat=server></{0}:Loading>")]
    [ToolboxBitmap(typeof(Loading), "Loading.bmp")]  
    [DesignerAttribute(typeof(LoadingDesigner), typeof(IDesigner))]
    public class Loading : WebControl
    {
        #region Private Variables

        private const string DEFAULT_LOADING_TEXT = "Loading...";
        private const string LOADING_IMAGE_VISIBILITY = "visible";
        private const string LOADING_IMAGE_HIDDEN = "hidden";
        private const string DEFAULT_TEXT_FONT = "Verdana";
        private const string DEFAULT_TEXT_SIZE = "Medium";
        private const string DEFAULT_TEXT_WEIGHT = "Normal";
        private const string DEFAULT_TEXT_COLOR = "Black";
        private const string DEFAULT_TEXT_STYLE = "Normal";
        private const string TEXT_STYLE_ITALIC = "Italic";
        private const string TEXT_WEIGHT_BOLD = "Bold";
        private const string TEXT_DECORATION_UNDERLINE = "Underline";
        private const string TEXT_DECORATION_OVERLINE = "Overline";
        private const string TEXT_DECORATION_STRIKEOUT = "Line-through";
        private const string DEFAULT_TEXT_DECORATION = "None";            
        		
        private string _LoadingText = DEFAULT_LOADING_TEXT;
        private string _LoadingImagePath = "";        
        private string _TextStyle = String.Empty;
        private string _IsLoadingImageVisible = LOADING_IMAGE_HIDDEN;

        private bool _IsAsyncPostBack = false;
        private Color DEFAULT_BACK_COLOR = Color.Transparent;

        private HAlignmentValues _HorizontalAlignment = HAlignmentValues.Center;
        private VAlignmentValues _VerticalAlignment = VAlignmentValues.Middle;
        private VAlignmentValues _TextVAlignment = VAlignmentValues.Middle;
        private RelativeTextLocation _TextLocation = RelativeTextLocation.TextAfterImage;                

        private string _UtilityScripts = "<SCRIPT LANGUAGE='JavaScript'>if(document.getElementById){var isLoaded = true;}	function ShowObject(obj){if (isLoaded){obj.style.display = 'block';}}	function HideObject(obj){if (isLoaded){obj.style.display = 'none';}}</SCRIPT>";        
        private string _LoadingDivTextAfterImage = "<div id='{0}' align=\"{1}\" style=\"vertical-align:{2}\" title=\"{9}\"><table width=\"100%\" height=\"100%\" bgcolor=\"{7}\"><tr><td align=\"{1}\" style=\"vertical-align:{2}\"><table><tr><td><img src=\"{3}\" style=\"vertical-align:middle;visibility:{8}\" /><label {5}>{6}</label></td></tr></table></td></tr></table></div>";
        private string _LoadingDivTextBeforeImage = "<div id='{0}' align=\"{1}\" style=\"vertical-align:{2}\" title=\"{9}\"><table width=\"100%\" height=\"100%\" bgcolor=\"{7}\"><tr><td align=\"{1}\" style=\"vertical-align:{2}\"><table><tr><td><label {5}>{6}</label><img src=\"{3}\" style=\"vertical-align:middle;visibility:{8}\" /></td></tr></table></td></tr></table></div>";
        private string _HideDiv = "<SCRIPT LANGUAGE='JavaScript'>	var divLoadObject = document.getElementById(\"{0}\");HideObject(divLoadObject);</script>";
        private string _TextStyleFormat = "style=\"font-family:{0}; font-size:{1}; font-weight:{2}; font-style:{3};text-decoration:{4};vertical-align:{5}; color:{6}\"";
        #endregion

        #region Properties
        /// <summary>
        /// Loading Text 
        /// </summary>
        [Description("Text to be displayed while loading")]
        [Bindable(true)]
        [Category("Appearance")]
        [DefaultValue("Loading...")]
        [Localizable(true)]        
        public string LoadingText
        {
            get { return _LoadingText; }
            set { _LoadingText = value; }
        }

        /// <summary>
        /// Loading Image path
        /// </summary>
        [Description("Image path of the image to be displayed while loading")]
        [UrlProperty]
        [Category("Appearance")]
        [Editor(typeof(UrlEditor), typeof(UITypeEditor))]
        public string LoadingImagePath
        {
            get { return _LoadingImagePath; }
            set { _LoadingImagePath = ResolveUrl(value); }
        }

        /// <summary>
        /// Controls Horizontal alignment
        /// </summary>
        [Description("Horizontal alignment of the control")]
        [DefaultValue("")]
        [Category("Appearance")]
        public HAlignmentValues HorizontalAlignment
        {
            get { return _HorizontalAlignment; }
            set { _HorizontalAlignment = value; }
        }

        /// <summary>
        /// Controls Vertical alignment
        /// </summary>
        [Description("Vertical alignment of the control")]
        [DefaultValue("")]
        [Category("Appearance")]
        public VAlignmentValues VerticalAlignment
        {
            get { return _VerticalAlignment; }
            set { _VerticalAlignment = value; }
        }

        /// <summary>
        /// Vertical Alignment of text in control
        /// </summary>
        [Description("Vertical alignment of the text in the control")]
        [DefaultValue("")]
        [Category("Appearance")]
        public VAlignmentValues TextVAlignment
        {
            get { return _TextVAlignment; }
            set { _TextVAlignment = value; }
        }

        /// <summary>
        /// Text relative location in control
        /// </summary>
        [Description("Location of the text relative to image in the control")]
        [DefaultValue("")]
        [Category("Appearance")]
        public RelativeTextLocation TextLocation
        {
            get { return _TextLocation; }
            set { _TextLocation = value; }
        }
        #endregion
                
        #region Constructor
        public Loading()
        {
            //Binding of event handlers for processing
            this.Init += new EventHandler(Loading_Init);
            this.Load += new EventHandler(Loading_Load);

            //Default values loaded
            this.BackColor = DEFAULT_BACK_COLOR;
            this._TextStyle = string.Format(_TextStyleFormat, DEFAULT_TEXT_FONT, DEFAULT_TEXT_SIZE, DEFAULT_TEXT_WEIGHT,DEFAULT_TEXT_STYLE, DEFAULT_TEXT_DECORATION, _TextVAlignment, DEFAULT_TEXT_COLOR);
        } 

        #endregion

        #region Private Methods

        /// <summary>
        /// Load method of the control
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Loading_Load(object sender, EventArgs e)
        {
            //Once the page has loaded, Loading control is made hidden
            if (!_IsAsyncPostBack)
            {
                WebControl tempWC = sender as WebControl;
                tempWC.Page.Response.Write(string.Format(_HideDiv, "loadingScreen"));
            }
        }

        /// <summary>
        /// Initialize method of the control
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Loading_Init(object sender, EventArgs e)
        {
            //Once the page life cycle starts, loading control is formaed and shown while page load ends
            string loadingDiv = string.Empty;

            // For making the control AJAX enabled
            ScriptManager currPageScriptManager = ScriptManager.GetCurrent(Page) as ScriptManager;
            if (currPageScriptManager != null)
                _IsAsyncPostBack = currPageScriptManager.IsInAsyncPostBack;
            if (!_IsAsyncPostBack)
            {
                WebControl tempWC = sender as WebControl;

                //Setting up of style as set for the control
                SetStyleAttributes();
                switch (_TextLocation)
                {                    
                    case RelativeTextLocation.TextBeforeImage:
                        loadingDiv = string.Format(_LoadingDivTextBeforeImage, "loadingScreen", _HorizontalAlignment, _VerticalAlignment, _LoadingImagePath, _LoadingText, _TextStyle, _LoadingText, GetColorCode(this.BackColor), _IsLoadingImageVisible, this.ToolTip);
                        break;
                    case RelativeTextLocation.TextAfterImage:
                    default:
                        loadingDiv = string.Format(_LoadingDivTextAfterImage, "loadingScreen", _HorizontalAlignment, _VerticalAlignment, _LoadingImagePath, _LoadingText, _TextStyle, _LoadingText, GetColorCode(this.BackColor), _IsLoadingImageVisible, this.ToolTip);
                        break;                       
                }

                //Loading control is displayed to the user while page loads
                tempWC.Page.Response.Write(_UtilityScripts + loadingDiv);
                tempWC.Page.Response.Flush();
            }
        }

        /// <summary>
        /// Style attributes of different control present in the div is handled/set here
        /// </summary>
        private void SetStyleAttributes()
        {
            // All visual style set is applied to the loading control
            if (String.IsNullOrEmpty(_LoadingImagePath))
                _IsLoadingImageVisible = LOADING_IMAGE_HIDDEN;
            else
                _IsLoadingImageVisible = LOADING_IMAGE_VISIBILITY;

            string textDecoration = DEFAULT_TEXT_DECORATION;
            if (this.Font.Overline || this.Font.Strikeout || this.Font.Underline)
            {
                textDecoration = "";
                textDecoration = this.Font.Overline ? textDecoration + TEXT_DECORATION_OVERLINE + " " : textDecoration;
                textDecoration = this.Font.Strikeout ? textDecoration + TEXT_DECORATION_STRIKEOUT + " " : textDecoration;
                textDecoration = this.Font.Underline ? textDecoration + TEXT_DECORATION_UNDERLINE : textDecoration;                
            }

            if (this.Font.ToString() != "" || this.Font.Bold || this.Font.Overline || this.Font.Strikeout || this.Font.Underline)
                _TextStyle = string.Format(_TextStyleFormat, string.IsNullOrEmpty(this.Font.Name) ? DEFAULT_TEXT_FONT : this.Font.Name, string.IsNullOrEmpty(this.Font.Size.ToString()) ? DEFAULT_TEXT_SIZE : this.Font.Size.ToString(), this.Font.Bold ? TEXT_WEIGHT_BOLD : DEFAULT_TEXT_WEIGHT, this.Font.Italic ? TEXT_STYLE_ITALIC : DEFAULT_TEXT_STYLE, textDecoration, _TextVAlignment, this.ForeColor.Name);
            else
                _TextStyle = string.Format(_TextStyleFormat, DEFAULT_TEXT_FONT, DEFAULT_TEXT_SIZE, DEFAULT_TEXT_WEIGHT, DEFAULT_TEXT_STYLE, DEFAULT_TEXT_DECORATION, _TextVAlignment, DEFAULT_TEXT_COLOR);
        }

        /// <summary>
        /// Method to get any color code in hex string format
        /// </summary>
        /// <param name="col"></param>
        /// <returns></returns>
        private static string GetColorCode(Color col)
        {            
            return "#" + col.R.ToString("x").PadLeft(2, '0') + col.G.ToString("x").PadLeft(2, '0') + col.B.ToString("x").PadLeft(2, '0');
        } 
        #endregion

        #region Enums

        /// <summary>
        /// Holizontal Alignment values for the control
        /// </summary>
        public enum HAlignmentValues
        {
            Center,
            Left,
            Right
        }

        /// <summary>
        /// Vertical Alignment values for the control
        /// </summary>
        public enum VAlignmentValues
        {
            Baseline,
            Bottom,
            Middle,
            Top
        }

        /// <summary>
        /// Text location relative to image
        /// </summary>
        public enum RelativeTextLocation
        {
            TextAfterImage,
            TextBeforeImage
        }

        #endregion
    }
}