using System.Web.UI;
using System.Web.UI.WebControls;
using System.IO;


namespace CodeProjectExample
{
	/// <summary>
	/// An extension to the ImageButton class that presently provides greyed-out images when disabled.
	/// </summary>
	[ToolboxData("<{0}:ImageButtonExtended runat=server></{0}:ImageButtonExtended>")]
	public class ImageButtonExtended : ImageButton
	{
	#region :: Member Functions ::

		/// <summary>
		/// Assigns the proper image url depending on enabled state.
		/// </summary>
		/// <remarks>
		/// This implementation, i.e. using this.ImageUrl as the base, ensures that changes made to the image url during the cycle will not be lost.
		/// </remarks>
		private void AssignImageUrl()
		{
			//If enabled we simply assign the enabled image url
			if (base.Enabled)
			{
				this.ImageUrl = FileUtil.RemovePostfix(this.ImageUrl, DISABLED_POSTFIX);
				return;
			}

			//Get the url of the disabled version of the image
			string disabledImageUrl = FileUtil.InjectPostfix(this.ImageUrl, DISABLED_POSTFIX, true);

			//If the current image url is already set to the proper value (which it will be if the button was initialised as disabled)
			//there is no need to go thorugh the rest
			if (this.ImageUrl != disabledImageUrl)
			{
				//See if the disabled version of the image exists and if not create it
				string disabledImageFile = this.Context.Server.MapPath(disabledImageUrl);
				if (!File.Exists(disabledImageFile))
				{
					string activeImageFile = this.Context.Server.MapPath(this.ImageUrl);
					var img = ImageUtil.MakeMonochrome(activeImageFile);
					img.Save(disabledImageFile);
				}

				this.ImageUrl = disabledImageUrl;
			}
		}

	#endregion
	//.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-
	#region :: Overrides ::

		protected override void OnPreRender(System.EventArgs e)
		{
			//Assign the proper image url if enabled state changed. 
			//This will always be the case if the button's enabled state is initialised to false, so its only a small optmization for when the button starts out enabled.
			if (_enabledChanged)
			{
				AssignImageUrl();
			}
			base.OnPreRender(e);
		}
		//--------------------------------------------------------------------------
		public override bool Enabled
		{
			get	{ return base.Enabled; }
			set
			{
				//Register changes to the enabled state. We cannot set the image url at this point since it may not yet have been assigned.
				if (base.Enabled != value)
				{
					_enabledChanged = true;
					base.Enabled = value;
				}
			}
		}
		private bool _enabledChanged;

	#endregion
	//.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-
	#region :: Class Members ::

		private const string DISABLED_POSTFIX = "_disabled";

	#endregion
		
	}
}
