using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Text;
using System.Drawing.Design;
using System.IO;
using System.Web;
using System.Web.UI;
using System.Web.UI.HtmlControls;
using System.Web.UI.WebControls;
using System.Web.UI.Design;
using AjaxControlToolkit;

[assembly: System.Web.UI.WebResource("SmumCounty.WebControls.ModalForm.js", "text/javascript")]

namespace SmumCounty.WebControls
{
    [ToolboxData("<{0}:ModalForm runat=server></{0}:ModalForm>")]
    public class ModalForm : Panel
    {
        #region Fields

        //Control property variables.
        private string _title = "";
        private string _contentCss = "";
        private string _headerCss = "";
        private string _closeImageCss = "";
        private string _closeImage = "";
        private string _closeRolloverImage = "";
        private string _backgroundCss = "";
        private bool _dropShadow = false;
        private string _okControlId = "";
        private string _cancelControlId = "";
        private string _onOkScript = "";
        private string _activateControlId = "";
        private string _extenderId = "";

        //ModalPopup extender associated with this panel.
        ModalPopupExtender _extender = null;

        //Popup control panel of the modalpopup extender.
        Panel _popup = null;

        private const string _modalFormInclude = "ModalForm";

        #endregion Fields

        #region Constructor

        public ModalForm() : base()
        {}

        #endregion Constructor

        #region Properties

        [Bindable(true)]
        [Category("Header Appearance")]
        [DefaultValue("")]
        [Themeable(false)]
        [Description("The title in the header of the form.")]
        public string Title
        {
            get { return _title; }
            set { _title = value; }
        }

        [Bindable(true)]
        [Category("Header Appearance")]
        [DefaultValue("")]
        [Editor(typeof(ImageUrlEditor), typeof(UITypeEditor))]
        [Description("The image to use as the close icon in the header.")]
        public string CloseImage
        {
            get { return _closeImage; }
            set { _closeImage = value; }
        }

        [Bindable(true)]
        [Category("Header Appearance")]
        [DefaultValue("")]
        [Editor(typeof(ImageUrlEditor), typeof(UITypeEditor))]
        [Description("The image to display in place of the close image when the mouse hovers over it.")]
        public string CloseRolloverImage
        {
            get { return _closeRolloverImage; }
            set { _closeRolloverImage = value; }
        }

        [Bindable(true)]
        [Category("Appearance")]
        [DefaultValue("")]
        [Description("Stylesheet class to apply to the content area.")]
        public string ContentCss
        {
            get { return _contentCss; }
            set { _contentCss = value; }
        }

        [Bindable(true)]
        [Category("Appearance")]
        [DefaultValue("")]
        [Description("Stylesheet class to apply to the header area.")]
        public string HeaderCss
        {
            get { return _headerCss; }
            set { _headerCss = value; }
        }

        [Bindable(true)]
        [Category("Appearance")]
        [DefaultValue("")]
        [Description("Stylesheet class to apply to the close image.")]
        public string CloseImageCss
        {
            get { return _closeImageCss; }
            set { _closeImageCss = value; }
        }

        [Bindable(true)]
        [Category("Appearance")]
        [DefaultValue("")]
        [Description("The stylesheet that describes that background that will fill the screen when the modal form is loaded.")]
        public string BackgroundCss
        {
            get { return _backgroundCss; }
            set 
            { 
                _backgroundCss = value;
                if (_extender != null)
                    _extender.BackgroundCssClass = value;
            }
        }

        [Bindable(true)]
        [Category("Appearance")]
        [DefaultValue(false)]
        [Description("Should a drop shadow appear behind the modal form.")]
        public bool DropShadow
        {
            get { return _dropShadow; }
            set 
            { 
                _dropShadow = value;
                if (_extender != null)
                    _extender.DropShadow = value;
            }
        }

        [Bindable(true)]
        [Category("Behavior")]
        [DefaultValue("")]
        [Description("The ID of the control that will signal a save of the modal form.")]
        public string OkControlId
        {
            get { return _okControlId; }
            set 
            { 
                _okControlId = value;
                if (_extender != null)
                    _extender.OkControlID = value;
            }
        }

        [Bindable(true)]
        [Category("Behavior")]
        [DefaultValue("")]
        [Description("The ID of the control that will cancel out of the modal form.")]
        public string CancelControlId
        {
            get { return _cancelControlId; }
            set 
            { 
                _cancelControlId = value;
                if (_extender != null)
                    _extender.CancelControlID = value;
            }
        }

        [Bindable(true)]
        [Category("Behavior")]
        [DefaultValue("")]
        [Description("The client side javascript function that should execute when the modal form is saved.")]
        public string OnOkScript
        {
            get { return _onOkScript; }
            set 
            { 
                _onOkScript = value;
                if (_extender != null)
                    _extender.OnOkScript = value;
            }
        }

        [Bindable(true)]
        [Category("Behavior")]
        [DefaultValue("")]
        [Description("The ID of the control that will activate the modal form.")]
        public string ActivateControlId
        {
            get { return _activateControlId; }
            set 
            { 
                _activateControlId = value;
                if (_extender != null)
                    _extender.TargetControlID = value;
            }
        }

        [Bindable(true)]
        [Category("Behavior")]
        [DefaultValue("")]
        [Description("The ID of the ModalPopupExtender control associated with this modal form.")]
        public string ExtenderId
        {
            get { return _extenderId; }
            set { _extenderId = value; }
        }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Force the creation of child controls now so that the ModalPopup extender is created on init.
        /// </summary>
        /// <param name="e"></param>
        protected override void OnInit(EventArgs e)
        {
            EnsureChildControls();

            base.OnInit(e);
        }


        /// <summary>
        /// Create the ModalPopup extender and add it as a child of this control.
        /// </summary>
        protected override void CreateChildControls()
        {
            base.CreateChildControls();

            // Create the extender and the popup control panel to act on.
            _extender = new ModalPopupExtender();
            _extender.ID = ID + "_ModalPopupExtender";

            _popup = new System.Web.UI.WebControls.Panel();
            _popup.ID = ID + "_Popup";

            //Pass on ModalPopup extender properties of this control to the extender.
            _extender.TargetControlID = ActivateControlId;
            _extender.BackgroundCssClass = BackgroundCss;
            _extender.CancelControlID = CancelControlId;
            _extender.OkControlID = OkControlId;
            _extender.DropShadow = DropShadow;
            _extender.OnOkScript = OnOkScript;
            _extender.PopupControlID = _popup.ID;

            //Pass on panel properties of this control to the popup panel.
            _popup.Width = this.Width;
            _popup.CssClass = this.CssClass;
            this.CssClass = "";

            if (!this.DesignMode)
            {
                Controls.AddAt(0, _popup);
                Controls.AddAt(0, _extender);
            }
        }


        /// <summary>
        /// Register javascript needed.
        /// </summary>
        /// <param name="e"></param>
        protected override void OnPreRender(EventArgs e)
        {
            if (!Page.ClientScript.IsClientScriptIncludeRegistered(this.GetType(), _modalFormInclude))
                Page.ClientScript.RegisterClientScriptInclude(this.GetType(), _modalFormInclude, Page.ClientScript.GetWebResourceUrl(this.GetType(), "SmumCounty.WebControls.ModalForm.js"));

            base.OnPreRender(e);
        }


        /// <summary>
        /// Render HTML for control.
        /// </summary>
        /// <param name="output"></param>
        protected override void Render(HtmlTextWriter output)
        {
            //Create the header and content panels and add them to the base panel.
            CreateContainerControls();

            base.Render(output);
        }
        
        /// <summary>
        /// Create the header and content panels and place the existing child controls in the content panel.
        /// </summary>
        private void CreateContainerControls()
        {
            Panel header = new Panel();
            header.ID = ID + "_Header";
            _extender.PopupDragHandleControlID = header.ID;
            header.CssClass = this.HeaderCss;

            Panel title = new Panel();

            //Add title text to title panel.
            LiteralControl titleText = new LiteralControl(this.Title);
            title.Controls.Add(titleText);

            //Place title on left of header panel.
            title.Style.Add("float", "left");
            header.Controls.Add(title);

            //Only create a close image panel if a close image has been specified.
            HtmlImage imageButton = null;
            if (CloseImage != "")
            {
                Panel closeImage = new Panel();

                //Add image to image panel.
                imageButton = new HtmlImage();
                imageButton.ID = ID + "_Button";
                imageButton.Border = 0;
                imageButton.Src = GetCloseImageUrl();
                imageButton.Style.Add("cursor", "pointer");

                //Both the image and the rollover image will be saved as attributes of the image to facilitate
                //rollover.
                imageButton.Attributes.Add("image", Path.GetFileName(CloseImage));
                imageButton.Attributes.Add("rolloverImage", Path.GetFileName(CloseRolloverImage));
                imageButton.Attributes.Add("onMouseOver", "RolloverImage(event);");
                imageButton.Attributes.Add("onMouseOut", "RolloverImage(event);");

                closeImage.Controls.Add(imageButton);

                //Place image on right of header panel.
                closeImage.Style.Add("float", "right");
                closeImage.CssClass = CloseImageCss;
                header.Controls.Add(closeImage);
            }

            Panel content = new Panel();
            content.CssClass = this.ContentCss;

            //Move all the children of the base panel to the children of the content panel.
            //Note as a child control is added to content, it is automatically removed from the base panel.
            int numChildren = this.Controls.Count;
            //Start at 2 since the popup panel and the modal popup extender will be 0 and 1.
            for (int i = 2; i < numChildren; i++)
                content.Controls.Add(this.Controls[2]);

            //Add the header and content panels to the popup panel.
            _popup.Controls.Add(header);
            _popup.Controls.Add(content);

            //Now that all controls are in their proper places, create the cancel javascript for the CancelControlId.
            if (CloseImage != "" && CancelControlId != "")
                imageButton.Attributes.Add("onClick", GetCloseScript());

            //Make sure the popup panel is hidden to begin.
            _popup.Style.Add("display", "none");
        }


        /// <summary>
        /// Get the javascript to close the modal form.
        /// </summary>
        /// <returns>
        /// The javascript which forces a click on the cancel control.
        /// </returns>
        private string GetCloseScript()
        {
            Control cancelControl = GetControl(this, CancelControlId);
            if (cancelControl != null)
                return String.Format("ModalFormClose(\"{0}\");", cancelControl.ClientID);
            else
                return "";
        }


        /// <summary>
        /// Perform a recursive search for the indicated control.
        /// </summary>
        /// <param name="root">
        /// The control whose children should be searched for the indicated control.
        /// </param>
        /// <param name="id">
        /// The id of the control to get.
        /// </param>
        /// <returns>
        /// The control with the indicated id.
        /// </returns>
        private Control GetControl(Control root, string id)
        {
            if (root.ID == id)
                return root;

            foreach (Control c in root.Controls)
            {
                Control t = GetControl(c, id);
                if (t != null)
                    return t;
            }

            return null;
        } 


        /// <summary>
        /// Get the url for the close image, taking into account any active theme.
        /// </summary>
        /// <returns>
        /// The url for the close image, taking into account any active theme.
        /// </returns>
        private string GetCloseImageUrl()
        {
            string image = CloseImage ;

            //Look for image in theme specific Images directory if a theme is in use.
            if (Page.Theme != "")
            {
                //Check to see if this file exists.
                if (File.Exists(Page.Request.MapPath("~") + "\\App_Themes\\" + Page.Theme + "\\Images\\" + image))
                    image = "~/App_Themes/" + Page.Theme + "/Images/" + image;
            }

            return image;
        }


        /// <summary>
        /// Show the modal form.
        /// </summary>
        public void Show()
        {
            //Do this by calling Show() on the associated modalpopup extender.
            _extender.Show();
        }


        /// <summary>
        /// Hide the modal form.
        /// </summary>
        public void Hide()
        {
            //Do this by calling Hide() on the associated modalpopup extender.
            _extender.Hide();
        }

        #endregion Methods
    }
}
