using System;
using System.Data;
using System.Configuration;
using System.Drawing;

namespace ImageTemplateNet
{
    /// <summary>
    /// Summary description for ElementConfig
    /// </summary>
    public abstract class ElementConfig 
    {
        private string m_Id;
        private Dimension m_Width;
        private Dimension m_Height;
        private int m_X;
        private int m_Y;
        private float m_Rotation;
        private Color m_BackgroundColor = Color.Transparent;
        private uint m_Duration;
        private ElementConfig m_Parent;

        public ElementConfig(ElementConfig parent)
        {
            this.m_Parent = parent;
        }


        /// <summary>
        /// How long the frame should show for.  Only applies to animated gifs
        /// </summary>
        public uint Duration
        {
            get
            {
                return m_Duration;
            }
            set
            {
                this.m_Duration = value;
            }
        }

        /// <summary>
        /// The X position the element should draw at
        /// </summary>
        public int X
        {
            get
            {
                return m_X;
            }
            set
            {
                this.m_X = value;
            }
        }

        /// <summary>
        /// The Y position the element should draw at
        /// </summary>
        public int Y
        {
            get
            {
                return m_Y;
            }
            set
            {
                this.m_Y = value;
            }
        }

        /// <summary>
        /// The Width of the element
        /// </summary>
        public string Width
        {
            get
            {
                return m_Width.ToString();
            }
            set
            {
                this.m_Width = new Dimension(value);
            }
        }

        /// <summary>
        /// The Height of the element
        /// </summary>
        public string Height
        {
            get
            {
                return m_Height.ToString();
            }
            set
            {
                this.m_Height = new Dimension(value);
            }
        }

        public Dimension WidthSetting
        {
            get
            {
                return m_Width;
            }
        }

        public Dimension HeightSetting
        {
            get
            {
                return m_Height;
            }
        }


        /// <summary>
        /// The background color of the element
        /// </summary>
        public Color BackgroundColor
        {
            get
            {
                return m_BackgroundColor;
            }
            set
            {
                this.m_BackgroundColor = value;
            }
        }

        /// <summary>
        /// The ID Of the element
        /// </summary>
        public string Id
        {
            get
            {
                return m_Id;
            }
            set
            {
                this.m_Id = value;
            }
        }

        /// <summary>
        /// The X,Y,Width and Height of the element as integers
        /// </summary>
        public Rectangle Bounds
        {
            get
            {
                return GetBounds(Size.Empty);
            }
        }

        public Rectangle GetBounds(Size imageSize)
        {
            int parentWidth = 0;
            int parentHeight = 0;
            if (m_Width.SizeType == SizeType.Pecentage || m_Height.SizeType == SizeType.Pecentage) {
                if (m_Parent != null)
                {
                    Rectangle parentBounds = m_Parent.Bounds;
                    parentWidth = parentBounds.Width;
                    parentHeight = parentBounds.Height;
                }
                else if (!imageSize.IsEmpty)
                {
                    parentWidth = imageSize.Width;
                    parentHeight = imageSize.Height;
                }
            }
            return new Rectangle(m_X, m_Y, m_Width.GetActualSize(parentWidth), m_Height.GetActualSize(parentHeight));
        }


        public Size Size
        {
            get
            {
                Rectangle rect = Bounds;
                return rect.Size;
            }
        }

        /// <summary>
        /// The X,Y,Width and Height of the element as floats
        /// </summary>
        public RectangleF FBounds
        {
            get
            {
                Rectangle rect = Bounds;
                return new RectangleF(m_X, m_Y, rect.Width, rect.Height);
            }
        }


        public ElementConfig ParentElement
        {
            get
            {
                return m_Parent;
            }
        }


        protected void CopyBaseProperties(ElementConfig config)
        {
            config.Id = Id;
            config.X = X;
            config.Y = Y;
            config.Duration = Duration;
            config.m_Width = WidthSetting;
            config.m_Height = HeightSetting;
            config.m_BackgroundColor = BackgroundColor;
        }

        #region ICloneable Members

        public abstract ElementConfig Clone();

        #endregion
    }
}