using System;
using System.ComponentModel;
using System.Text;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;

namespace OnlineScorekeeper
{
	/// <summary>Creates a Google calendar reminder hyperlink control with appropriate url</summary>
	[DefaultProperty("Text")]
	[ToolboxData("<{0}:GoogleCalendarEventLink runat=server ></{0}:GoogleCalendarEventLink>")]
	public class GoogleCalendarEventLink : HyperLink
	{
		/// <summary>Date and time for the start of the event.</summary>
		public DateTime StartDateTime
		{
			get
			{
				DateTime d = (DateTime)ViewState["StartDateTime"];
				return d;
			}
			set
			{
				ViewState["StartDateTime"] = value;
			}
		}

		/// <summary>Date and time when the event ends</summary>
		public DateTime EndDateTime
		{
			get
			{
				DateTime d = (DateTime)ViewState["EndDateTime"];
				return d;
			}
			set
			{
				ViewState["EndDateTime"] = value;
			}
		}

		/// <summary>Specifies if this event is an all day event that has no start and end times</summary>
		public bool AllDayEvent
		{
			get
			{
				bool b = false;
				if (ViewState["AllDayEvent"] != null)
				{
					b = (bool)ViewState["AllDayEvent"];
				}
				return b;
			}
			set
			{
				ViewState["AllDayEvent"] = value;
			}
		}

		/// <summary>Title or summary of the event</summary>
		public string EventTitle
		{
			get
			{

				String s = (string)ViewState["EventTitle"];
				return s;
			}
			set
			{
				ViewState["EventTitle"] = value;
			}
		}

		/// <summary>Location of the event, typically an address or company name and city, state combination</summary>
		public string EventLocation
		{
			get
			{
				string s = (string)ViewState["EventLocation"];
				return ((s == null) ? string.Empty : s);
			}

			set
			{
				ViewState["EventLocation"] = value;
			}
		}

		/// <summary>Description of the event.  Simple html is allowed</summary>
		public string EventDescription
		{
			get
			{
				String s = (String)ViewState["EventDescription"];
				return ((s == null) ? String.Empty : s);
			}

			set
			{
				ViewState["EventDescription"] = value;
			}
		}

		/// <summary>Website of event's organization</summary>
		/// <remarks>Do not include http://</remarks>
		public string OrganizerWebsite
		{
			get
			{
				String s = (String)ViewState["OrganizerWebsite"];
				return ((s == null) ? String.Empty : s);
			}

			set
			{
				ViewState["OrganizerWebsite"] = value;
			}
		}

		/// <summary>Name of event's organizer</summary>
		public string OrganizerName
		{
			get
			{
				String s = (String)ViewState["OrganizerName"];
				return ((s == null) ? String.Empty : s);
			}

			set
			{
				ViewState["OrganizerName"] = value;
			}
		}

		/// <summary>Tells Google Calendar whether to show this event as busy or free in the user's calendar</summary>
		/// <remarks>Can be overridden by the user</remarks>
		public bool MarkAsBusy
		{
			get
			{
				bool b = false;
				if (ViewState["MarkAsBusy"] != null)
				{
					b = (bool)ViewState["MarkAsBusy"];
				}
				return b;
			}
			set
			{
				ViewState["MarkAsBusy"] = value;
			}
		}

		/// <summary>Sets a default Google Calendar image as the hyperlink's image</summary>
		/// <remarks>Setting this property will override the ImageUrl property</remarks>
		public GoogleButtonStyle ButtonStyle
		{
			get
			{
				GoogleButtonStyle s = (GoogleButtonStyle)ViewState["ButtonStyle"];
				return ((ViewState["ButtonStyle"] == null) ? GoogleButtonStyle.None : s);

			}
			set
			{
				if (value == GoogleButtonStyle.None)
				{
					this.ImageUrl = null;
				}
				else
				{
					if (value == GoogleButtonStyle.FavIcon)
					{
						this.ImageUrl = "http://www.google.com/calendar/images/favicon.ico";
					}
					else
					{
						this.ImageUrl = "http://www.google.com/calendar/images/ext/" + value.ToString().ToLower() + ".gif";
					}
				}
				ViewState["ButtonStyle"] = value;
			}
		}

		protected override void Render(HtmlTextWriter writer)
		{
			StringBuilder url = new StringBuilder();
			url.Append("http://www.google.com/calendar/event?");
			url.Append("action=TEMPLATE");

			// Event title
			string eventText = this.EventTitle;
			if (string.IsNullOrEmpty(eventText))
			{
				eventText = this.Text;
			}
			if (!string.IsNullOrEmpty(eventText))
			{
				url.AppendFormat("&text={0}", HttpUtility.UrlEncode(eventText));
			}

			// Event dates
			// TODO: Validate that a start or end date has been specified
			url.Append("&dates=");
			if (this.StartDateTime != null)
			{
				if (this.AllDayEvent || (this.StartDateTime == this.EndDateTime))
				{
					url.AppendFormat("{0}/{0}", this.StartDateTime.ToString("yyyyMMdd"));
				}
				else
				{
					// TODO: Validate that EndDateTime is set, because this is not an all day event
					const string UTCFORMATSTRING = "yyyyMMdd\\THHmmss\\Z";
					url.AppendFormat("{0}/{1}", this.StartDateTime.ToUniversalTime().ToString(UTCFORMATSTRING), this.EndDateTime.ToUniversalTime().ToString(UTCFORMATSTRING));
				}
			}

			// TODO: Apparently on sprop is required by google, so validate that one is specified
			// Organization info
			if (!string.IsNullOrEmpty(this.OrganizerName))
			{
				url.AppendFormat("&sprop=name:{0}", HttpUtility.UrlEncode(this.OrganizerName));
			}
			if (!string.IsNullOrEmpty(this.OrganizerWebsite))
			{
				url.AppendFormat("&sprop=website:{0}", HttpUtility.UrlEncode(this.OrganizerWebsite));
			}

			// Event location
			if (!string.IsNullOrEmpty(this.EventLocation))
			{
				url.AppendFormat("&location={0}", HttpUtility.UrlEncode(this.EventLocation));
			}

			// Event description
			if (!string.IsNullOrEmpty(this.EventDescription))
			{
				url.AppendFormat("&details={0}", HttpUtility.UrlEncode(this.EventDescription));
			}

			// Free/Busy
			// Only add to url if true since default false and url could already be really long
			if (this.MarkAsBusy)
			{
				url.AppendFormat("&trp={0}", this.MarkAsBusy);
			}

			// Set the NavigateUrl
			this.NavigateUrl = url.ToString();
			base.Render(writer);
		}
	}

	/// <summary>List of button images available from Google.com</summary>
	/// <remarks>See http://www.google.com/googlecalendar/event_publisher_guide.html for list of buttons</remarks>
	public enum GoogleButtonStyle
	{
		/// <summary>Do not use a default google image</summary>
		None,
		///<summary>Small button with no hint text</summary>
		/// <remarks>http://www.google.com/calendar/images/ext/gc_button1.gif</remarks>
		GC_Button1,
		///<summary>Medium button with hint text</summary>
		/// <remarks>http://www.google.com/calendar/images/ext/gc_button2.gif</remarks>
		GC_Button2,
		///<summary>Medium button with no hint text</summary>
		/// <remarks>http://www.google.com/calendar/images/ext/gc_button3.gif</remarks>
		GC_Button3,
		///<summary>Large button with hint text</summary>
		/// <remarks>http://www.google.com/calendar/images/ext/gc_button4.gif</remarks>
		GC_Button4,
		///<summary>Large button with no hint text</summary>
		/// <remarks>http://www.google.com/calendar/images/ext/gc_button5.gif</remarks>
		GC_Button5,
		///<summary>Google Calendar's FavIcon</summary>
		/// <remarks>
		/// Caution: Might not work with all browsers
		///	http://www.google.com/calendar/images/favicon.ico
		///	</remarks>
		FavIcon
	}
}

