using System;

namespace CustomWebControls
{
    [Serializable]
    public struct DateRange
    {
        public static readonly DateRange EMPTY = new DateRange();
        
        readonly DateTime from;
        readonly DateTime to;


        public DateRange(DateTime from, DateTime to)
        {
            this.from = from;
            this.to = to;
        }


        public DateTime From
        {
            get { return from; }
        }

        public DateTime To
        {
            get { return to; }
        }

        public TimeSpan TimeSpan
        {
            get
            {
                return to - from;
            }
        }
        
        public bool Contains(DateTime time)
        {
            return from <= time && time < to;
        }

        public DateRange Include(DateRange otherRange)
        {
            return Include(otherRange.From).Include(otherRange.To);
        }

        public DateRange Include(DateTime date)
        {
            if (date < from)
                return new DateRange(date, to);
            else if (date > to)
                return new DateRange(from, date);
            else 
                return this;
        }

        /// <summary>
        /// Creates a one day (24 hr) long DateRange starting at DateTime
        /// </summary>
        public static DateRange CreateDay(DateTime dateTime){
            return new DateRange(dateTime, dateTime.AddDays(1));
        }

        #region operators and overrides
        public override int GetHashCode()
        {
            return from.GetHashCode() + 29*to.GetHashCode();
        }

        public override bool Equals(object obj)
        {
            if (ReferenceEquals(this, obj)) return true;
            if (!(obj is DateRange)) return false;
            DateRange dateRange = (DateRange) obj;
            if (!Equals(from, dateRange.from)) return false;
            if (!Equals(to, dateRange.to)) return false;
            return true;
        }


        public static bool operator == (DateRange d1, DateRange d2)
        {
            return d1.Equals(d2);
        }

        public static bool operator !=(DateRange d1, DateRange d2)
        {
            return !d1.Equals(d2);
        }
        #endregion

    }
}
