Imports System.ComponentModel
Imports System.Web.UI.WebControls
Imports System.Collections.Generic

''' <summary>
''' ListBoxList control that auto-populates from an Enum type
''' </summary>
''' <remarks>
''' Set the EnumType property to the FULL class name of the Enum type you wish
''' to use. For example for DayOfWeek enum, set it to System.DayOfWeek
''' 
''' If your enum is defined inside a class, you must use the nested operator +
''' e.g. for MyEnum defined inside MyClass, which is in namespace MyNamespace
''' use "MyNamespace.MyClass+MyEnum"
''' 
''' To control how the text is shown, use can use FixNames = True to amend underscores 
''' CamelSpacing, e.g. 'A_Test' becomes 'A Test' and 'MyValue' becomes 'My Value'
''' 
''' For precise name control use the System.ComponentModel.DescriptionAttribute
''' on each value. This is used by default if found.
''' 
''' If XmlEnumAttributes are present you can use these by setting UseXmlEnumNames = True
''' </remarks>
<System.Web.UI.ToolboxData("<{0}:ListBoxEnum runat=""server"" EnumType="""" />")> _
Public Class ListBoxEnum
    Inherits ListBox

#Region "Properties"

    ''' <summary>
    ''' The type of enum to display
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks>
    ''' For global enums use 'Namespace.EnumName'
    ''' For enums inside classes use 'Namespace.ClassName+EnumName'
    ''' e.g. 'System.DayOfWeek', 'MyApp.CarClass+CarTypes'
    ''' </remarks>
    <Bindable(True), Category("Behavior")> _
    Public Property EnumType() As String
        Get
            Return _enumType
        End Get
        Set(ByVal value As String)
            _enumType = value
        End Set
    End Property
    Private _enumType As String = Nothing

    ''' <summary>
    ''' Fix enum names by changing underscores to spaces
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    <Bindable(True), Category("Behavior"), DefaultValue(GetType(System.Boolean), "False")> _
    Public Property FixNames() As Boolean
        Get
            Return _fixNames
        End Get
        Set(ByVal value As Boolean)
            _fixNames = value
        End Set
    End Property
    Private _fixNames As Boolean = False

    ''' <summary>
    ''' Use the XmlEnumAttribute name if present
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    <Bindable(True), Category("Behavior"), DefaultValue(GetType(System.Boolean), "False")> _
    Public Property UseXmlEnumNames() As Boolean
        Get
            Return _useXmlEnumNames
        End Get
        Set(ByVal value As Boolean)
            _useXmlEnumNames = value
        End Set
    End Property
    Private _useXmlEnumNames As Boolean = False

#End Region

#Region "Event overrides"

    Protected Overrides Sub OnInit(ByVal e As System.EventArgs)
        MyBase.OnInit(e)
        'if not in design mode, fill the list
        If Me.DesignMode = False Then Me.FillRadioList()
    End Sub


#End Region

#Region "Internal functions"

    ''' <summary>
    ''' Fill the list with the values and names from the enum
    ''' </summary>
    ''' <remarks>
    ''' If the enum type is invalid it shows the error as an disable item
    ''' </remarks>
    Private Sub FillRadioList()
        'Remove existing items
        Me.Items.Clear()

        'Check for empty value
        If String.IsNullOrEmpty(EnumType) Then
            Me.Items.Add(New ListItem("No EnumType specified in " & Me.ID, "", False))
            Return
        End If

        'set items from values
        Try
            Dim values As Dictionary(Of Integer, String) = _
                    EnumHelper.GetEnumerationValues(EnumType, FixNames, UseXmlEnumNames)
            For Each value As Integer In values.Keys
                Me.Items.Add(New ListItem(values.Item(value), value.ToString))
            Next

        Catch ex As EnumHelper.InvalidEnumTypeException
            'handle problem with the EnumType
            Me.Items.Add(New ListItem(ex.Message, "", False))
        End Try
    End Sub


#End Region


End Class


