Imports System.Reflection
Imports System.Collections.Generic

''' <summary>
''' Helper class to read enum values
''' </summary>
''' <remarks></remarks>
Public Class EnumHelper

    ''' <summary>
    ''' Exception for EnumType values
    ''' </summary>
    ''' <remarks></remarks>
    Public Class InvalidEnumTypeException
        Inherits ApplicationException
        Sub New(ByVal message As String)
            MyBase.New(message)
        End Sub
    End Class

    ''' <summary>
    ''' Return a dictionary of value-name pairs for Enum
    ''' </summary>
    ''' <param name="typename"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Shared Function GetEnumerationValues( _
                                ByVal typename As String, _
                                ByVal fixNames As Boolean, _
                                ByVal useXmlNames As Boolean) As Dictionary(Of Integer, String)
        'Get the type
        Dim t As Type = Type.GetType(typename, False, False)
        If t Is Nothing Then _
            Throw New InvalidEnumTypeException("Failed to find type " & typename)

        'Check that the type is based on Enum
        If Not t.IsSubclassOf(GetType(System.Enum)) Then _
            Throw New InvalidEnumTypeException("Type " & typename & " is not an enum")

        'Get the enum values
        Dim allValues() As Integer = CType(System.Enum.GetValues(t), Integer())
        'Get list of names 
        Dim allNames() As String = GetEnumNames(t, fixNames, useXmlNames)
        'Add values to result
        Dim result As New System.Collections.Generic.Dictionary(Of Integer, String)
        For i As Integer = 0 To allValues.Length - 1
            result.Add(allValues(i), allNames(i))
        Next
        Return result
    End Function

    ''' <summary>
    ''' Helper function that returns an array of names for Enum values
    ''' </summary>
    ''' <param name="t"></param>
    ''' <param name="fixNames">Set true if you want underscores converted to spaces and CamelCase converted to 'Camel Case'</param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Private Shared Function GetEnumNames(ByVal t As Type, ByVal fixNames As Boolean, ByVal useXML As Boolean) As String()
        'Set up result
        Dim result As New List(Of String)
        'Enum names (text) are defined as fields for the type
        For Each enumValue As FieldInfo In t.GetFields(BindingFlags.Public Or BindingFlags.Static)
            'If XML is specified and found, use this name
            If useXML Then
                Dim xmlAttr As System.Xml.Serialization.XmlEnumAttribute = _
                                    CType(Attribute.GetCustomAttribute(enumValue, _
                                    GetType(System.Xml.Serialization.XmlEnumAttribute)), _
                                     System.Xml.Serialization.XmlEnumAttribute)
                If xmlAttr IsNot Nothing Then
                    result.Add(xmlAttr.Name)
                    Continue For
                End If
            End If
            'Attempt to use the Description attribute (if present)
            Dim description As System.ComponentModel.DescriptionAttribute = _
                                CType(Attribute.GetCustomAttribute(enumValue, _
                                GetType(System.ComponentModel.DescriptionAttribute)), _
                                System.ComponentModel.DescriptionAttribute)
            If description IsNot Nothing Then
                'use this value
                result.Add(description.Description)
            Else
                'If not found use the Enum name, with adjustment if requested
                If fixNames Then
                    result.Add(AdjustName(enumValue.Name))
                Else
                    result.Add(enumValue.Name)
                End If
            End If

        Next
        Return result.ToArray
    End Function

    ''' <summary>
    ''' Helper function to allow name adjustment
    ''' </summary>
    ''' <param name="name">Name to adjust</param>
    ''' <returns>Amended name</returns>
    ''' <remarks>
    ''' </remarks>
    Private Shared Function AdjustName(ByVal name As String) As String
        Return AdjustCamelCase(name.Replace("_", " ")).Replace("  ", " ")
    End Function

    ''' <summary>
    ''' Helper function to add spaces to CamelCase names
    ''' </summary>
    ''' <param name="name"></param>
    ''' <returns>Name with spaces added</returns>
    ''' <remarks>
    ''' Converts values such as 'ThisIsATest' to 'This Is A Test'
    ''' </remarks>
    Private Shared Function AdjustCamelCase(ByVal name As String) As String
        Const ToTitleCase As String = "(\B[A-Z])"
        Return System.Text.RegularExpressions.Regex.Replace(name, ToTitleCase, " $1")
    End Function

End Class

