#region Licence
// Copyright (c) 2007 James Gregory (james@jagregory.com)
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without modification,
// are permitted provided that the following conditions are met:
// 
//     * Redistributions of source code must retain the above copyright notice,
//     this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright notice,
//     this list of conditions and the following disclaimer in the documentation
//     and/or other materials provided with the distribution.
//     * Neither the name of James Gregory nor the names of its
//     contributors may be used to endorse or promote products derived from this
//     software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
// OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#endregion

using System;
using System.Web.UI.WebControls;
using NUnit.Framework;
using Rhino.Mocks;
using Rhino.Mocks.Constraints;

namespace JAGregory.Controls.Tests
{
	[TestFixture]
	public class DeleGridTests
	{
		[Test]
		public void AlwaysRequestTotalDefaultsToFalse()
		{
			Assert.IsFalse(new DeleGrid().AlwaysRequestTotal, "AlwaysRequestTotal should default to false.");
		}

		[Test]
		public void AlwaysRequestsTotalReturnsSetValue()
		{
			DeleGrid grid = new DeleGrid();

			grid.AlwaysRequestTotal = true;

			Assert.IsTrue(grid.AlwaysRequestTotal, "AlwaysRequestsTotal value should be true after being set to true.");
		}

		[Test]
		public void SortingFieldShouldDefaultToEmptyString()
		{
			Assert.AreEqual(string.Empty, new DeleGrid().SortingField, "SortingField should default to an empty string.");
		}

		[Test]
		public void SortingFieldReturnsSetValue()
		{
			DeleGrid grid = new DeleGrid();
			
			grid.SortingField = "field";

			Assert.AreEqual("field", grid.SortingField, "SortingField should return value after being set.");
		}

		[Test]
		public void SortingDirectionShouldDefaultToAscending()
		{
			Assert.AreEqual(SortDirection.Ascending, new DeleGrid().SortingDirection, "SortingDirection should default to Ascending.");
		}

		[Test]
		public void SortingDirectionReturnsSetValue()
		{
			DeleGrid grid = new DeleGrid();

			grid.SortingDirection = SortDirection.Descending;

			Assert.AreEqual(SortDirection.Descending, grid.SortingDirection, "SortingDirection value should return value after being set.");
		}

		[Test]
		public void PageIndexReturnsSetValue()
		{
			DeleGrid grid = new DeleGrid();

			grid.PageIndex = 10;

			Assert.AreEqual(10, grid.PageIndex, "PageIndex should return value after being set.");
		}

		[Test]
		public void DataBindCallsPageDataRequest()
		{
			DeleGrid grid = new DeleGrid();
			bool called = false;

			grid.PageDataRequest += delegate {
				called = true;
				return new string[] {};
			};
			grid.DataBind();

			Assert.IsTrue(called, "PageDataRequest event should be raised on DataBind.");
		}

		[Test, ExpectedException(typeof(EventNotHandledException), ExpectedMessage = "The DeleGrid 'gridID' fired event PageDataRequest which wasn't handled.")]
		public void DataBindThrowsExceptionIfNoEventHandlerAttachedToPageDataRequest()
		{
			DeleGrid grid = new DeleGrid();
			
			grid.ID = "gridID";
			grid.TotalRecordCountRequest += delegate { return 10; };
			grid.DataBind();
		}

		[Test, ExpectedException(typeof(EventNotHandledException), ExpectedMessage = "The DeleGrid 'gridID' fired event TotalRecordCountRequest which wasn't handled.")]
		public void DataBindThrowsExceptionIfNoEventHandlerAttachedToTotalRecordCountRequestAndRecordsGreaterThanPageSize()
		{
			DeleGrid grid = new DeleGrid();

			grid.ID = "gridID";
			grid.AllowPaging = true;
			grid.PageSize = 2;
			grid.PageDataRequest += delegate { return new int[] { 1, 2, 3, 4 }; };
			grid.DataBind();
		}

		[Test]
		public void DataBindCallsTotalRecordCountRequest()
		{
			DeleGrid grid = new DeleGrid();
			bool called = false;

			grid.ID = "gridID";
			grid.AllowPaging = true;
			grid.PageSize = 2;
			grid.PageDataRequest += delegate { return new int[] { 1, 2, 3, 4 }; };
			grid.TotalRecordCountRequest += delegate {
				called = true;
				return 10;
			};

			try
			{
				grid.DataBind();
			}
			catch (NullReferenceException)
			{
				// this is something to-do with the internals of the datagrid, let it happen
				// as long as our method has been called first
			}

			Assert.IsTrue(called, "TotalRecordCountRequest should be called if records greater than page size.");
		}

		[Test]
		public void PagerConfigureDataSourceCalledOnDataBind()
		{
			MockRepository mocks = new MockRepository();
			IPager pager = mocks.DynamicMock<IPager>();
			ISorter sorter = mocks.Stub<ISorter>();
			int total = 101;

			using (mocks.Record())
			{
				pager.ConfigureDataSource(null, 0);
				LastCall.Constraints(Is.NotNull(), Is.Equal(total));
			}

			using (mocks.Playback())
			{
				DeleGrid grid = new DeleGrid(pager, sorter);

				grid.AllowPaging = true;
				grid.PageSize = 2;
				grid.PageDataRequest += delegate { return new int[] {1, 2, 3, 4}; };
				grid.TotalRecordCountRequest += delegate { return total; };

				try
				{
					grid.DataBind();
				}
				catch (NullReferenceException)
				{
					// this is something to-do with the internals of the datagrid, let it happen
					// as long as our method has been called first
				}
			}
		}

		[Test]
		public void SortingCallsSorterStore()
		{
			MockRepository mocks = new MockRepository();
			IPager pager = mocks.Stub<IPager>();
			ISorter sorter = mocks.DynamicMock<ISorter>();
			string expression = "field";

			using (mocks.Record())
			{
				sorter.Store(expression);
				LastCall.Repeat.Once();
			}

			using (mocks.Playback())
			{
				DeleGrid grid = new DeleGrid(pager, sorter);

				grid.TotalRecordCountRequest += delegate { return 10; };
				grid.PageDataRequest += delegate { return new string[] {}; };

				grid.Sort(expression, SortDirection.Ascending);
			}
		}
	}
}