using System;
using System.Web.UI.WebControls;
using System.Web;
using System.Web.UI;
using System.Collections;

namespace AddCssToDataGridSample
{
	/// <summary>
	/// Contains helpermethods for adding CSS classes and attributes to a datagrid
	/// </summary>
	
	/**** History	******
	 Version 1.0
	 ...contained the following methods:

	 - AddCssToDataGrid
	 - AddAttributesToRows
	 - AddAttributesToCells
	 - AddConfirm
	 
	
	Version 1.1
	...changes made to 
	- AddCssToDataGrid
	so it now also adds CSS to buttons in the datagrid.
	
	
	Version 1.2
	...changes made to
	- The entire class, so it now implements
	the IHttpModule interface.
	This allows the class to work with *no code*
	and just find DataGrids in all webpages served
	from an Asp.Net application and add css classes to them.
	**********************/
	

	public class DataGridHelper : IHttpModule
	{
		/// <summary>
		/// This method should be called
		/// before the datagrid is databound.
		/// This method will make sure that each row
		/// (DataGridItem), cell and button in the DataGrid 
		/// has a CSS class added with the name 
		/// of the ItemType.
		/// E.g."Item_Row", "Header_Cell",
		/// "Button_Delete", etc.
		/// </summary>
		/// <param name="aGrid">The DataGrid to add CSS to</param>
		public static void AddCssToDataGrid(DataGrid aGrid)
		{
			aGrid.ItemDataBound +=new DataGridItemEventHandler(AddCssToDataGridItem);
		}


		/// <summary>
		/// This method is used to add cssClass names 
		/// to rows in a DataGrid. 
		/// It is used by the AddCssToDataGrid method,
		/// and is not intended for use by users.
		/// The AddCssToDataGrid adds this method to the 
		/// ItemDataBound event  of a datagrid,
		/// and lets this method handle adding
		/// cssclass names to the cells of rows.
		/// </summary>
		public static void AddCssToDataGridItem(object sender, System.Web.UI.WebControls.DataGridItemEventArgs e)
		{
			//get the type of the row
			string type = e.Item.ItemType.ToString();

			//for all cells in the row...
			foreach(TableCell cell in e.Item.Cells)
			{
				//add the cells cssclass
				cell.Attributes.Add("class", type + "_cell");

				// *** Version 1.1	***********************
				//	Added CSS class names to buttons too.
				//	Thanks to R.vd.B for suggesting this : )
				// *****************************************
				
				//iterate over the controls in the cell
				foreach (WebControl control in cell.Controls)
				{
					//if it is a DataGridLinkButton
					if(control is LinkButton)
					{
						//typecast to LinkButton
						LinkButton theLinkButton = (LinkButton) control;
						//and add the right CssClass
						theLinkButton.CssClass = "Button_" + theLinkButton.CommandName; 
					}
					
						//if it is a Button
					else if(control is Button)
					{
						//typecast to Button
						Button theButton = (Button) control;
						//and add the right CssClass
						theButton.CssClass = "Button_" + theButton.CommandName; 
					}
				}		
			}
			//add the rows cssclass 
			e.Item.Attributes.Add("class", type + "_row");
		}


		/// <summary>
		/// This method adds an attribute to all cells
		/// of the Item and AlternatingItems in a DataGrid
		/// </summary>
		/// <param name="theGrid">The DataGrid whose cells 
		/// to add the attribute to</param>
		/// <param name="attributeKey">The name of the attribute, e.g. "title" or "onmouseover"</param>
		/// <param name="attributeValue">The value of the attribute
		///	e.g. "showmessage();" or "blue"</param>
		public static void AddAttributesToCells(DataGrid theGrid, string attributeKey, string attributeValue)
		{
			//For each row in the DataGrid (Items and AlternatingItems)
			foreach(DataGridItem item in theGrid.Items)
				//for each cell in the row
				foreach(TableCell cell in item.Cells)
					//add the attribute to the cell
					cell.Attributes.Add(attributeKey, attributeValue);
		}


		/// <summary>
		/// This method adds a specified attribute and value
		/// to all rows in a DataGrid
		/// </summary>
		/// <param name="theGrid">The DataGrid whose rows 
		/// to add the attribute to</param>
		/// <param name="attributeKey">The name of the attribute, e.g. "title" or "onmouseover"</param>
		/// <param name="attributeValue">The value of the attribute
		///	e.g. "showmessage();" or "blue"</param>
		public static void AddAttributesToRows(DataGrid theGrid, string attributeKey, string attributeValue)
		{
			foreach(DataGridItem item in theGrid.Items)
				item.Attributes.Add(attributeKey, attributeValue);
		}


		/// <summary>
		/// This method adds a javascript confirmation
		/// dialog to a button in a datagrid column.
		/// For example a "Delete" button.
		/// </summary>
		/// <param name="theGrid">The grid containing the row to add confirmation dialogs to</param>
		/// <param name="columnNumber">The number of the column containing the linkbuttons</param>
		/// <param name="question"></param>
		public static void AddConfirm(DataGrid theGrid, int columnNumber, string question)
		{
			//for each row in the DataGrid
			foreach(DataGridItem item in theGrid.Items)
			{
				//for each webcontrol in the column
				foreach(WebControl control in item.Cells[columnNumber].Controls)
					//if it is a LinkButton or Button
				{
						if(control is LinkButton  || control is Button)
					 //add the attribute
					 control.Attributes.Add("OnClick", " javascript:return confirm('" + question + "');");
				}
			}
		}

		#region IHttpModule Members
		/*********************************************
		 * These two methods are added to satisfy the 
		 * requirements of the IHttpModule interface.
		 * 
		 * The Init method adds a subscribermethod:
		 * "context_PreRequestHandlerExecute" to the
		 * PreRequestHandlerExecute event, thereby
		 * getting access to the controls of the Page
		 * and adding the CSS classes to the datagrid.
		 * 
		 * *******************************************/

		/// <summary>
		/// Adds an EventHandler to the PreRequestHandlerExecute event of the current Application
		/// </summary>
		/// <param name="context">The HttpApplication to modify</param>
		public void Init(HttpApplication context)
		{
			context.PreRequestHandlerExecute += new EventHandler(context_PreRequestHandlerExecute);
		}


		/// <summary>
		/// Only added to satisfy the IHttpModule interface requirements
		/// </summary>
		public void Dispose(){}

		#endregion


		/// <summary>
		/// Adds an EventHandler to the current Page objects Init event
		/// </summary>
		/// <param name="sender">The current page being processed</param>
		/// <param name="e">Not used</param>
		private void context_PreRequestHandlerExecute(object sender, EventArgs e)
		{
			//we typecast the sender object as an IHttpHandler
			IHttpHandler handler = ((HttpApplication)sender).Context.Handler;
			//then we check to see if this is a Page
			Page thePage = handler as Page;
			//if it is...
			if(thePage != null)
				//then we add an EventHandler to the Init event
				thePage.Init +=new EventHandler(thePage_Init);
		}


		/// <summary>
		/// Recursively searches for controls of a given type and returns them as an array of Control.
		/// </summary>
		/// <param name="theControl">The control to look in, e.g. a Page object or Panel, etc.</param>
		/// <param name="theTypeToLookFor">The type of Control to look for, e.g. DataGrid, CheckBox, etc.</param>
		/// <param name="theList">An arraylist to use while searching, to gather results in</param>
		/// <returns>An array containing all the controls found of the type to search for</returns>
		public  Control[] GetControls(Control theControl, Type theTypeToLookFor, ArrayList theList)
		{
			//if the control we are checking is of the given type
			if(theControl.GetType() == theTypeToLookFor)
				//then we add it to the list
				theList.Add(theControl);

			//if there are any childcontrols in this control
			if(theControl.Controls.Count > 0)
				//then we call this method again on all the childcontrols
				foreach(Control childControl in theControl.Controls)
					//and add the returned array to the ArrayList
					theList.AddRange(GetControls(childControl, theTypeToLookFor, theList));
			
			//return the found controls (if any) as an array of Control
			return (Control[]) theList.ToArray(typeof(Control));
		}


		/// <summary>
		/// Checks the Page for any DataGrid objects and adds CSS to them
		/// </summary>
		/// <param name="sender">The Page object</param>
		/// <param name="e">Not used</param>
		private void thePage_Init(object sender, EventArgs e)
		{
			//declare a new arraylist to hold the datagrids found on the page
			ArrayList theList = new ArrayList();
			//get a reference to the page object
			Page thePage = (Page)sender;
			//find all DataGrid objects on the page
			Control[] dataGrids = GetControls(thePage, typeof(DataGrid), theList);
			//for each of them - add an eventhandler to their ItemDataBound event 
			//using the AddCssToDataGrid method
			foreach(Control aControl in dataGrids)
				AddCssToDataGrid((DataGrid)aControl);
		}
	}
}	