using System;
using System.ComponentModel;
using System.Text;
using System.Drawing;
using System.Web.UI;
using System.Web.UI.WebControls;

namespace Altinoren.HTMLControls 
{

	/// <summary>
	/// Creates a button on the page that users can click to show or hide a control on the page.
	/// </summary>
	/// <example>
	/// The following example demonstrates how to use a ShowHideButton control in a page.
	/// <code>
	/// <![CDATA[
	/// <%@ Register TagPrefix="ahc" Namespace="Altinoren.HTMLControls" Assembly="Altinoren.HTMLControls.ShowHideButton" %>
	/// <html>
	///	<head>
	///	</head>
	///	<body>
	///		<h3>ShowHideButton Sample</h3><br>
	///		<asp:label id="Label1" runat="server">Click on the button to hide this text:</asp:label><br>
	///		<ahc:ShowHideButton
	///			id="ShowHideButton1"
	///			runat="server"
	///			InitialState="Hidden"
	///			ControlToShowHide="Label1"
	///			AlternativeText="Show it again"
	///			Text="Hide" />
	///       
	///	</body>
	/// </html>
	/// ]]>
	/// </code>
	/// </example>
	/// <seealso cref="System.Web.UI.WebControls.Button"/>
	[Designer(typeof(Altinoren.HTMLControls.Design.ShowHideButtonDesigner)),
	ToolboxData("<{0}:ShowHideButton runat=server></{0}:ShowHideButton>"),
	ToolboxBitmap(typeof(Bitmap))
	]
	public class ShowHideButton : Button 
	{

		/// <summary>
		/// State types for the element that this button controls.
		/// </summary>
		public enum states
		{
			/// <summary>
			/// Specifies that the <see cref="ControlToShowHide"/> will be
			/// hidden when the page first loads.
			/// </summary>
			Hidden = 0,
			/// <summary>
			/// Specifies that the <see cref="ControlToShowHide"/> will be visible
			/// when the page first loads. This is the default value.
			/// </summary>
			Visible,
		}

		/// <summary>
		/// Initializes a new instance of the <see cref="ShowHideButton"/> class.
		/// </summary>
		public ShowHideButton(): base() 
		{
			InitialState = states.Visible;
		}
	
		/// <summary>
		/// The name of the control to show or hide.
		/// </summary>
		/// <remarks>
		/// Only html elements with <b>runat="server"</b> attribute are handled.
		/// </remarks>
		[
		Bindable(true),
		Category("Attributes"),
		Description("The name of the control to show or hide."),
		DefaultValue("")
		]
		public String ControlToShowHide 
		{
			get 
			{
				object savedState;
                
				savedState = this.ViewState["ControlToShowHide"];
				if (savedState != null) 
				{
					return (String) savedState;
				}
				return "";
			}
			set 
			{ 
				ViewState["ControlToShowHide"] = value; 
			}
		}

		/// <summary>
		/// Alternative text to be displayed when the button is in <b>Hidden</b> state.
		/// </summary>
		/// <remarks>
		/// If omitted, <see cref="System.Web.UI.WebControls.Button.Text"/> property will be used instead.
		/// <para>
		/// Remember that when the <see cref="states"/> property is set to <b>Hidden</b>,
		/// this string will be displayed on the button when the page first loads.
		/// </para>
		/// </remarks>
		[
		Bindable(true),
		Category("Attributes"),
		Description("Alternative text to be displayed when the button is in Hidden state."),
		DefaultValue("")
		]
		public String AlternativeText 
		{
			get 
			{
				object savedState;
                
				savedState = this.ViewState["AlternativeText"];
				if (savedState != null) 
				{
					return (String) savedState;
				}
				return "";
			}
			set 
			{ 
				ViewState["AlternativeText"] = value; 
			}
		}

		/// <summary>
		/// Starting <see cref="states"/> of the control that the button controls.
		/// </summary>
		/// <remarks>
		/// Remember that when this property is set to <b>Hidden</b>,
		/// <see cref="AlternativeText"/> will be displayed on the button when the page first loads.
		/// </remarks>
		[
		Bindable(true),
		Category("Attributes"),
		Description("Starting state of the element that the button controls."),
		DefaultValue(states.Visible)
		]
		public states InitialState 
		{
			get 
			{
				object savedState;
                
				savedState = this.ViewState["InitialState"];
				if (savedState != null) 
				{
					return (states) savedState;
				}
				return states.Visible;
			}
			set 
			{ 
				ViewState["InitialState"] = value; 
			}
		}

		/// <summary>
		/// Gets or sets a value indicating whether the control retains it's <see cref="states"/> after the postback.
		/// </summary>
		/// <remarks>
		/// During a postback, the control will revert it's <see cref="states"/> to the initial value regardless of the last state on the client.
		/// If you set this property to <b>True</b>, the <b>Visible</b> or <b>Hidden</b> state of the target control
		/// will retain the last value set on the client side.
		/// </remarks>
		[
		Bindable(true),
		Category("Attributes"),
		Description("Gets or sets a value indicating whether the control retains it's state after the postback."),
		DefaultValue(false)
		]
		public bool PersistState
		{
			get 
			{
				object savedState;
                
				savedState = this.ViewState["PersistState"];
				if (savedState != null) 
				{
					return (bool) savedState;
				}
				return false;
			}
			set 
			{ 
				ViewState["PersistState"] = value; 
			}
		}

		protected override void OnPreRender( EventArgs e ) 
		{
			// Script to apply InitialState to handled control when the page loads
			if(!Page.IsStartupScriptRegistered("ShowHideButton_Startup"))
			{
				StringBuilder scriptString = new StringBuilder(1000);
				scriptString.Append("\n<script language=JavaScript>\n");
				scriptString.Append("	var i;\n");
				scriptString.Append("	for (i = 0; i < ShowHideButtons_State.length; i++) {\n");
				scriptString.Append("		if (ShowHideButtons_State[i] != 'Visible') {\n");
				scriptString.Append("			if (document.all) {\n");
				scriptString.Append("				showHideButton(document.all[ShowHideButtons[i]], true);\n");
				scriptString.Append("			}\n");
				scriptString.Append("			else if (document.getElementById) {\n");
				scriptString.Append("				showHideButton(document.getElementById(ShowHideButtons[i]), true);\n");
				scriptString.Append("			}\n");
				scriptString.Append("		}\n");
				scriptString.Append("	}\n");
				scriptString.Append("</script>\n");

				Page.RegisterStartupScript("ShowHideButton_Startup", scriptString.ToString());
			}

			// Main function script
			if(!Page.IsClientScriptBlockRegistered("ShowHideButton_Script"))
			{
				StringBuilder scriptString = new StringBuilder(1000);
				scriptString.Append("\n<script language=JavaScript>\nfunction showHideButton(sourceCtl, isStart) {\n");
				scriptString.Append("	if (typeof(ShowHideButtons) == 'undefined') return;\n");
				scriptString.Append("	if (typeof(ShowHideButtons_Ctl) == 'undefined') return;\n");
				scriptString.Append("	if (typeof(ShowHideButtons_Text) == 'undefined') return;\n");
				scriptString.Append("	if (typeof(ShowHideButtons_AText) == 'undefined') return;\n");
				scriptString.Append("	var i, bType;\n");
				scriptString.Append("	if (document.all) {\n");
				scriptString.Append("		for (i = 0; i < ShowHideButtons.length; i++) {\n");
				scriptString.Append("			if (typeof(document.all[ShowHideButtons_Ctl[i]]) != 'undefined') {\n");
				scriptString.Append("				if (document.all[ShowHideButtons[i]] == sourceCtl) {\n");
				scriptString.Append("					if (document.all[ShowHideButtons_Ctl[i]].style.display == 'inline' || document.all[ShowHideButtons_Ctl[i]].style.display == '') {\n");
				scriptString.Append("						document.all[ShowHideButtons_Ctl[i]].style.display = 'none';\n");
				scriptString.Append("						sourceCtl.value = ShowHideButtons_AText[i];\n");
				scriptString.Append("						document.all[ShowHideButtons[i] + '_CState'].value = 'Hidden';\n");
				scriptString.Append("					} else {\n");
				scriptString.Append("						document.all[ShowHideButtons_Ctl[i]].style.display = 'inline';\n");
				scriptString.Append("						sourceCtl.value = ShowHideButtons_Text[i];\n");
				scriptString.Append("						document.all[ShowHideButtons[i] + '_CState'].value = 'Visible';\n");
				scriptString.Append("					}\n");
				scriptString.Append("				}\n");
				scriptString.Append("			}\n");
				scriptString.Append("		}\n");
				scriptString.Append("	}\n");
				scriptString.Append("	else if (document.getElementById) {\n");
				scriptString.Append("		for (i = 0; i < ShowHideButtons.length; i++) {\n");
				scriptString.Append("			if (typeof(document.getElementById(ShowHideButtons_Ctl[i])) != 'undefined') {\n");
				scriptString.Append("				if (document.getElementById(ShowHideButtons[i]) == sourceCtl) {\n");
				scriptString.Append("					if (document.getElementById(ShowHideButtons_Ctl[i]).style.display == 'inline') {\n");
				scriptString.Append("						document.getElementById(ShowHideButtons_Ctl[i]).style.display = 'none';\n");
				scriptString.Append("						sourceCtl.value = ShowHideButtons_AText[i];\n");
				scriptString.Append("						document.getElementById(ShowHideButtons[i] + '_CState').value = 'Hidden';\n");
				scriptString.Append("					} else if (document.getElementById(ShowHideButtons_Ctl[i]).style.display == 'none') {\n");
				scriptString.Append("						document.getElementById(ShowHideButtons_Ctl[i]).style.display = 'inline';\n");
				scriptString.Append("						sourceCtl.value = ShowHideButtons_Text[i];\n");
				scriptString.Append("						document.getElementById(ShowHideButtons[i] + '_CState').value = 'Visible';\n");
				scriptString.Append("					} else {\n");
				scriptString.Append("						document.getElementById(ShowHideButtons_Ctl[i]).style.display = 'none';\n");
				scriptString.Append("						sourceCtl.value = ShowHideButtons_AText[i];\n");
				scriptString.Append("						document.getElementById(ShowHideButtons[i] + '_CState').value = 'Hidden';\n");
				scriptString.Append("					}\n");
				scriptString.Append("				}\n");
				scriptString.Append("			}\n");
				scriptString.Append("		}\n");
				scriptString.Append("	}\n");
				scriptString.Append("	if (!isStart) {\n");
				scriptString.Append("		window.event.cancelBubble = true;\n");
				scriptString.Append("		window.event.returnValue = false;\n");
				scriptString.Append("	}\n");
				scriptString.Append("}\n");
				scriptString.Append("</script>\n");

				Page.RegisterClientScriptBlock("ShowHideButton_Script", scriptString.ToString());
			}

			if (ControlToShowHide != "")
			{
				Control vControl = NamingContainer.FindControl(ControlToShowHide);
				if (vControl == null)
					throw new InvalidOperationException("ControlToShowHide: '" + ControlToShowHide + "' is not found in the NamingContainer.");

				// Array declarations to be used by showHideButton() JavaScript function.
				Page.RegisterArrayDeclaration("ShowHideButtons", "'" + ClientID + "'" );
				Page.RegisterArrayDeclaration("ShowHideButtons_Ctl", "'" + vControl.ClientID + "'" );
				Page.RegisterArrayDeclaration("ShowHideButtons_Text", "'" + this.Text + "'" );

				// If AlternativeText is not set, Text property will be used for both.
				if (AlternativeText != "")
				{
					Page.RegisterArrayDeclaration("ShowHideButtons_AText", "'" + this.AlternativeText + "'" );
				}
				else
				{
					Page.RegisterArrayDeclaration("ShowHideButtons_AText", "'" + this.Text + "'" );
				}

				// Take PersistState into account if the page is a postback
				if (Page.IsPostBack)
				{
					// If PersistState is true, use hidden 'ClientID'_CState form element's value
					// as starting state of the control.
					if (PersistState)
					{
						object clientState;
						clientState = Page.Request.Params.Get(ClientID + "_CState");

						if (clientState != null)
						{
							Page.RegisterArrayDeclaration("ShowHideButtons_State", "'" + clientState.ToString() + "'" );
						}
						else
						{
							Page.RegisterArrayDeclaration("ShowHideButtons_State", "'" + this.InitialState.ToString() + "'" );
						}
					}
					else
					{
						Page.RegisterArrayDeclaration("ShowHideButtons_State", "'" + this.InitialState.ToString() + "'" );
					}
				}
				else
				{
					Page.RegisterArrayDeclaration("ShowHideButtons_State", "'" + this.InitialState.ToString() + "'" );
				}
			}
				
			base.OnPreRender( e );
		}

		// Adds a hidden field to hold the client side state of the control for persisting
		//state during a postback if PersistState is true.
		protected override void Render(HtmlTextWriter output)
		{
			if (Page.IsPostBack)
			{
				if (PersistState)
				{
					object clientState;
					clientState = Page.Request.Params.Get(ClientID + "_CState");
					
					if (clientState != null)
					{
						output.Write("<input type=\"hidden\" name=\"" + ClientID + "_CState\" value=\"" + clientState.ToString() + "\">");
					}
					else
					{
						output.Write("<input type=\"hidden\" name=\"" + ClientID + "_CState\" value=\"" + this.InitialState.ToString() + "\">");
					}
				}
				else
				{
					output.Write("<input type=\"hidden\" name=\"" + ClientID + "_CState\" value=\"" + this.InitialState.ToString() + "\">");
				}
			}
			else
			{
				output.Write("<input type=\"hidden\" name=\"" + ClientID + "_CState\" value=\"" + this.InitialState.ToString() + "\">");
			}

			base.Render(output);
		}

		// Adds HTML attributes and styles that need to be rendered to the
		// specified System.Web.UI.HtmlTextWriter.
		protected override void AddAttributesToRender( HtmlTextWriter writer ) 
		{
			//HACK: Add our attributes before calling the base. This way, our onclick
			//event is written before the validator's onclick, if any. Then our
			//type attribute as "button" is written before base's type "submit".
			//In fact, this is not the proper way to do it and may have side effects.
			//A better way would be deriving the class from something else, not button.
			writer.AddAttribute(HtmlTextWriterAttribute.Onclick.ToString(), GetAttributes(), true );
			writer.AddAttribute(HtmlTextWriterAttribute.Type.ToString(), "button", true );
			base.AddAttributesToRender( writer );
		}
		
		internal String GetAttributes()
		{
			StringBuilder Options = new StringBuilder();

			Options.Append ("javascript:showHideButton(this);return false");

			return Options.ToString();
		}


	}

}