using System;
using System.ComponentModel;
using System.Text;
using System.Drawing;
using System.Web.UI;
using System.Web.UI.WebControls;

namespace Altinoren.HTMLControls.BrowserWindow {

	/// <summary>
	/// Creates a link on the page that users can click to browse to another page in a new window.
	/// </summary>
	/// <remarks>
	/// Use the BWHyperLink to open a new window to another web page. Set the <see cref="NavigateURL"/> to specify the page to be displayed.
	/// </remarks>
	/// <example>
	/// The following example demonstrates how to use a BWHyperLink control to open another Web page in a new window.
	/// <code>
	/// <![CDATA[
	/// <html>
	///	<head>
	///	</head>
	///	<body>
	///		<h3>BWHyperLink Sample</h3>
	///		Click on the link:<br>
	///		<abw:BWHyperLink id="BWHyperLink1"
	///			 Text="This"
	///			 NavigateUrl="http://altinoren.com"
	///			 Left=10
	///			 Top=10
	///			 WindowWidth=400
	///			 WindowHeight=400
	///			 Location=False
	///			 MenuBar=False
	///			 Resizable=False
	///			 ScrollBars=False
	///			 Status=False
	///			 ToolBar=False
	///			 runat="server"/> link will open a new window.
	///       
	///	</body>
	/// </html>
	/// ]]>
	/// </code>
	/// </example>
	/// <seealso cref="System.Web.UI.WebControls.HyperLink"/>
	[Designer(typeof(Altinoren.HTMLControls.BrowserWindow.BWHyperLinkDesigner)),
		ToolboxData("<{0}:BWHyperLink runat=server></{0}:BWHyperLink>"),
		ToolboxBitmap(typeof(Bitmap), "BWHyperLink.bmp")
	]
	public class BWHyperLink : System.Web.UI.WebControls.HyperLink {

		/// <summary>
		/// Initializes a new instance of the <see cref="BWHyperLink"/> class.
		/// </summary>
		public BWHyperLink(): base() {
			
			Random ran = new Random(this.GetHashCode());
			WindowName = "BWHyperLink_"+ ran.Next().ToString();
			
			base.Target = "_blank";
		}
	
		/// <summary>
		/// Specifies the window name to refer to the new window programmatically.
		/// </summary>
		/// <remarks>
		/// You may set the WindowName to a predefined string or leave it blank.
		/// In the letter case, a WindowName with "BWHyperLink_" plus a random string will be assigned by default.
		/// </remarks>
		[
			Bindable(true),
			Category("Attributes"),
			Description("Specifies the window name to refer the new window programmatically."),
			DefaultValue("")
		]
		public String WindowName {
			get {
				object savedState;
                
				savedState = this.ViewState["WindowName"];
				if (savedState != null) {
					return (String) savedState;
				}
				return "";
			}
            set { 
				ViewState["WindowName"] = value; 
			}
		}

		/// <summary>
		/// Specifies whether to display the input field for entering URLs directly into the browser. The default is yes.
		/// </summary>
		[
		Bindable(true),
		Category("Window Appearance"),
		Description("Specifies whether to display the input field for entering URLs directly into the browser. The default is yes."),
		DefaultValue(true)
		]
		public Boolean Location
		{
			get 
			{ 
				object savedState;
                
				savedState = this.ViewState["Location"];
				if (savedState != null) 
				{
					return (Boolean) savedState;
				}
				return true;
			}
			set { ViewState["Location"] = value; }
		}
		
		/// <summary>
		/// Specifies whether to add a status bar at the bottom of the new window. The default is yes.
		/// </summary>
		[
		Bindable(true),
		Category("Window Appearance"),
		Description("Specifies whether to add a status bar at the bottom of the new window. The default is yes."),
		DefaultValue(true)
		]
		public Boolean Status
		{
			get 
			{ 
				object savedState;
                
				savedState = this.ViewState["Status"];
				if (savedState != null) 
				{
					return (Boolean) savedState;
				}
				return true;
			}
			set { ViewState["Status"] = value; }
		}
		
		/// <summary>
		/// Specifies whether to display the menu bar. The default is yes.
		/// </summary>
		[
		Bindable(true),
		Category("Window Appearance"),
		Description("Specifies whether to display the menu bar. The default is yes."),
		DefaultValue(true)
		]
		public Boolean MenuBar 
		{
			get 
			{ 
				object savedState;
                
				savedState = this.ViewState["MenuBar"];
				if (savedState != null) 
				{
					return (Boolean) savedState;
				}
				return true;
			}
			set { ViewState["MenuBar"] = value; }
		}
		
		/// <summary>
		/// Specifies whether to display the browser toolbar, making buttons such as Back, Forward, and Stop available. The default is yes.
		/// </summary>
		[
		Bindable(true),
		Category("Window Appearance"),
		Description("Specifies whether to display the browser toolbar, making buttons such as Back, Forward, and Stop available. The default is yes."),
		DefaultValue(true)
		]
		public Boolean ToolBar 
		{
			get 
			{ 
				object savedState;
                
				savedState = this.ViewState["ToolBar"];
				if (savedState != null) 
				{
					return (Boolean) savedState;
				}
				return true;
			}
			set { ViewState["ToolBar"] = value; }
		}

		/// <summary>
		/// Specifies whether to display horizontal and vertical scroll bars. The default is yes.
		/// </summary>
		[
		Bindable(true),
		Category("Window Appearance"),
		Description("Specifies whether to display horizontal and vertical scroll bars. The default is yes."),
		DefaultValue(true)
		]
		public Boolean ScrollBars 
		{
			get 
			{ 
				object savedState;
                
				savedState = this.ViewState["ScrollBars"];
				if (savedState != null) 
				{
					return (Boolean) savedState;
				}
				return true;
			}
			set { ViewState["ScrollBars"] = value; }
		}

		/// <summary>
		/// Specifies whether to display resize handles at the corners of the new window. The default is yes.
		/// </summary>
		[
		Bindable(true),
		Category("Window Appearance"),
		Description("Specifies whether to display resize handles at the corners of the new window. The default is yes."),
		DefaultValue(true)
		]
		public Boolean Resizable {
			get { 
				object savedState;
                
				savedState = this.ViewState["Resizable"];
				if (savedState != null) {
					return (Boolean) savedState;
				}
				return true;
			}
			set { ViewState["Resizable"] = value; }
		}

		/// <summary>
		/// Gets or sets the top position, in pixels. This value is relative to the upper-left corner of the screen.
		/// </summary>
		[
		Bindable(true),
		Category("Window Layout"),
		Description("Gets or sets the top position, in pixels. This value is relative to the upper-left corner of the screen."),
		DefaultValue("")
		]
		public Unit Top {
			get {
				object savedState;
                
				savedState = this.ViewState["Top"];
				if (savedState != null) {
					return (Unit) savedState;
				}
				return Unit.Empty;
			}
			set { 
				ViewState["Top"] = value; 
			}
		}

		/// <summary>
		/// Gets or sets the left position, in pixels. This value is relative to the upper-left corner of the screen.
		/// </summary>
		[
		Bindable(true),
		Category("Window Layout"),
		Description("Gets or sets the left position, in pixels. This value is relative to the upper-left corner of the screen."),
		DefaultValue("")
		]
		public Unit Left {
			get {
				object savedState;
                
				savedState = this.ViewState["Left"];
				if (savedState != null) {
					return (Unit) savedState;
				}
				return Unit.Empty;
			}
			set { 
				ViewState["Left"] = value; 
			}
		}

		/// <summary>
		/// Specifies the height of the window, in pixels.
		/// </summary>
		[
		Bindable(true),
		Category("Window Layout"),
		Description("Gets or sets the height of the window, in pixels."),
		DefaultValue("")
		]
		public Unit WindowHeight {
			get {
				object savedState;
                
				savedState = this.ViewState["Height"];
				if (savedState != null) {
					return (Unit) savedState;
				}
				return Unit.Empty;
			}
			set { 
				ViewState["Height"] = value; 
			}
		}

		/// <summary>
		/// Gets or sets the width of the window, in pixels.
		/// </summary>
		[
		Bindable(true),
		Category("Window Layout"),
		Description("Gets or sets the width of the window, in pixels."),
		DefaultValue("")
		]
		public Unit WindowWidth {
			get {
				object savedState;
                
				savedState = this.ViewState["Width"];
				if (savedState != null) {
					return (Unit) savedState;
				}
				return Unit.Empty;
			}
			set { 
				ViewState["Width"] = value; 
			}
		}

		protected override void AddAttributesToRender( HtmlTextWriter writer ) {
			base.AddAttributesToRender( writer );
			writer.AddAttribute("onclick", GetAttributes(), true );
		}
		
		internal String GetAttributes() 
		{
			StringBuilder Options = new StringBuilder();

			Options.Append ("javascript:window.open( '");
			Options.Append (this.ResolveUrl(NavigateUrl));
			Options.Append ("', '");
			Options.Append (WindowName);
			Options.Append ("', 'menubar=");
			
			// MenuBar
			if ( MenuBar ) 
			{
				Options.Append( "1," );
			}
			else 
			{
				Options.Append( "0," );
			}

			// ToolBar
			Options.Append ("toolbar=");
			if ( ToolBar ) 
			{
				Options.Append( "1," );
			}
			else 
			{
				Options.Append( "0," );
			}

			// Status
			Options.Append ("status=");
			if ( Status ) 
			{
				Options.Append( "1," );
			}
			else 
			{
				Options.Append( "0," );
			}

			// ScrollBar
			//
			// Old: ("scrollbar=")
			// Bug fixed. Thank you Max.
			Options.Append ("scrollbars=");
			if ( ScrollBars ) 
			{
				Options.Append( "1," );
			}
			else 
			{
				Options.Append( "0," );
			}
			
			// Location
			Options.Append ("location=");
			if ( Location ) 
			{
				Options.Append( "1," );
			}
			else 
			{
				Options.Append( "0," );
			}

			// Resizable
			Options.Append ("resizable=");
			if ( Resizable ) 
			{
				Options.Append( "1," );
			}
			else 
			{
				Options.Append( "0," );
			}

			if ( Top != Unit.Empty ) 
			{
				Options.Append ("top=");
				Options.Append ( Top.Value.ToString() );
				Options.Append (",");
			}
			
			if ( Left != Unit.Empty ) 
			{
				Options.Append ("left=");
				Options.Append ( Left.Value.ToString() );
				Options.Append (",");
			}	

			if ( WindowHeight != Unit.Empty ) 
			{
				Options.Append ("height=");
				Options.Append ( WindowHeight.Value.ToString() );
				Options.Append (",");
			}	

			if ( WindowWidth != Unit.Empty ) 
			{
				Options.Append ("width=");
				Options.Append ( WindowWidth.Value.ToString() );
			}	

			Options.Append( "'); return false;" );

			return Options.ToString();
		}


	}

}