using System;
using System.ComponentModel;
using System.Text;
using System.Drawing;
using System.Web.UI;
using System.Web.UI.WebControls;

namespace Altinoren.HTMLControls.BrowserWindow {

	/// <summary>
	/// Creates a button on the page that users can click to browse to another page in a new window.
	/// </summary>
	/// <remarks>
	/// Use the BWButton to open a new window to another web page. Set the <see cref="NavigateURL"/> to specify the page to be displayed.
	/// </remarks>
	/// <example>
	/// The following example demonstrates how to use a BWButton control to open another Web page in a new window.
	/// <code>
	/// <![CDATA[
	/// <html>
	///	<head>
	///	</head>
	///	<body>
	///		<h3>BWButton Sample</h3>
	///		Click on the button:<br>
	///		<abw:BWButton id="BWButton1"
	///			 Text="This"
	///			 NavigateUrl="http://altinoren.com"
	///			 Left=10
	///			 Top=10
	///			 WindowWidth=400
	///			 WindowHeight=400
	///			 Location=False
	///			 MenuBar=False
	///			 Resizable=False
	///			 ScrollBars=False
	///			 Status=False
	///			 ToolBar=False
	///			 runat="server"/> button will open a new window.
	///       
	///	</body>
	/// </html>
	/// ]]>
	/// </code>
	/// </example>
	/// <seealso cref="System.Web.UI.WebControls.Button"/>
	[Designer(typeof(Altinoren.HTMLControls.BrowserWindow.BWButtonDesigner)),
		ToolboxData("<{0}:BWButton runat=server></{0}:BWButton>"),
		ToolboxBitmap(typeof(Bitmap), "BWButton.bmp")
	]
	public class BWButton : System.Web.UI.WebControls.Button {

		/// <summary>
		/// Initializes a new instance of the <see cref="BWButton"/> class.
		/// </summary>
		public BWButton(): base() {
			
			Random ran = new Random(this.GetHashCode());
			WindowName = "BWButton_"+ ran.Next().ToString();
		}

		/// <summary>
		/// The URL to navigate to.
		/// </summary>
		/// <remarks>
		/// Use the <b>NavigateUrl</b> property to specify the URL to navigate to in the new window when the
		/// BWButton control is clicked.
		/// </remarks>
		[
		Bindable(true),
		Category("Navigation"),
		Description("The URL to naviagte to."),
		DefaultValue("")
		]
		public String NavigateUrl 
		{
			get 
			{
				object savedState;
                
				savedState = this.ViewState["NavigateUrl"];
				if (savedState != null) 
				{
					return (String) savedState;
				}
				return "";
			}
			set 
			{ 
				ViewState["NavigateUrl"] = value; 
			}
		}
	
		/// <summary>
		/// Specifies the window name to refer to the new window programmatically.
		/// </summary>
		/// <remarks>
		/// You may set the WindowName to a predefined string or leave it blank.
		/// In the letter case, a WindowName with "BWButton_" plus a random string will be assigned by default.
		/// </remarks>
		[
			Bindable(true),
			Category("Attributes"),
			Description("Specifies the window name to refer the new window programmatically."),
			DefaultValue("")
		]
		public String WindowName {
			get {
				object savedState;
                
				savedState = this.ViewState["WindowName"];
				if (savedState != null) {
					return (String) savedState;
				}
				return "";
			}
            set { 
				ViewState["WindowName"] = value; 
			}
		}

		/// <summary>
		/// Specifies whether to display the input field for entering URLs directly into the browser. The default is yes.
		/// </summary>
		[
		Bindable(true),
		Category("Window Appearance"),
		Description("Specifies whether to display the input field for entering URLs directly into the browser. The default is yes."),
		DefaultValue(true)
		]
		public Boolean Location
		{
			get 
			{ 
				object savedState;
                
				savedState = this.ViewState["Location"];
				if (savedState != null) 
				{
					return (Boolean) savedState;
				}
				return true;
			}
			set { ViewState["Location"] = value; }
		}
		
		/// <summary>
		/// Specifies whether to add a status bar at the bottom of the new window. The default is yes.
		/// </summary>
		[
		Bindable(true),
		Category("Window Appearance"),
		Description("Specifies whether to add a status bar at the bottom of the new window. The default is yes."),
		DefaultValue(true)
		]
		public Boolean Status
		{
			get 
			{ 
				object savedState;
                
				savedState = this.ViewState["Status"];
				if (savedState != null) 
				{
					return (Boolean) savedState;
				}
				return true;
			}
			set { ViewState["Status"] = value; }
		}
		
		/// <summary>
		/// Specifies whether to display the menu bar. The default is yes.
		/// </summary>
		[
		Bindable(true),
		Category("Window Appearance"),
		Description("Specifies whether to display the menu bar. The default is yes."),
		DefaultValue(true)
		]
		public Boolean MenuBar 
		{
			get 
			{ 
				object savedState;
                
				savedState = this.ViewState["MenuBar"];
				if (savedState != null) 
				{
					return (Boolean) savedState;
				}
				return true;
			}
			set { ViewState["MenuBar"] = value; }
		}
		
		/// <summary>
		/// Specifies whether to display the browser toolbar, making buttons such as Back, Forward, and Stop available. The default is yes.
		/// </summary>
		[
		Bindable(true),
		Category("Window Appearance"),
		Description("Specifies whether to display the browser toolbar, making buttons such as Back, Forward, and Stop available. The default is yes."),
		DefaultValue(true)
		]
		public Boolean ToolBar 
		{
			get 
			{ 
				object savedState;
                
				savedState = this.ViewState["ToolBar"];
				if (savedState != null) 
				{
					return (Boolean) savedState;
				}
				return true;
			}
			set { ViewState["ToolBar"] = value; }
		}

		/// <summary>
		/// Specifies whether to display horizontal and vertical scroll bars. The default is yes.
		/// </summary>
		[
		Bindable(true),
		Category("Window Appearance"),
		Description("Specifies whether to display horizontal and vertical scroll bars. The default is yes."),
		DefaultValue(true)
		]
		public Boolean ScrollBars 
		{
			get 
			{ 
				object savedState;
                
				savedState = this.ViewState["ScrollBars"];
				if (savedState != null) 
				{
					return (Boolean) savedState;
				}
				return true;
			}
			set { ViewState["ScrollBars"] = value; }
		}

		/// <summary>
		/// Specifies whether to display resize handles at the corners of the new window. The default is yes.
		/// </summary>
		[
		Bindable(true),
		Category("Window Appearance"),
		Description("Specifies whether to display resize handles at the corners of the new window. The default is yes."),
		DefaultValue(true)
		]
		public Boolean Resizable {
			get { 
				object savedState;
                
				savedState = this.ViewState["Resizable"];
				if (savedState != null) {
					return (Boolean) savedState;
				}
				return true;
			}
			set { ViewState["Resizable"] = value; }
		}

		/// <summary>
		/// Gets or sets the top position, in pixels. This value is relative to the upper-left corner of the screen.
		/// </summary>
		[
		Bindable(true),
		Category("Window Layout"),
		Description("Gets or sets the top position, in pixels. This value is relative to the upper-left corner of the screen."),
		DefaultValue("")
		]
		public Unit Top {
			get {
				object savedState;
                
				savedState = this.ViewState["Top"];
				if (savedState != null) {
					return (Unit) savedState;
				}
				return Unit.Empty;
			}
			set { 
				ViewState["Top"] = value; 
			}
		}

		/// <summary>
		/// Gets or sets the left position, in pixels. This value is relative to the upper-left corner of the screen.
		/// </summary>
		[
		Bindable(true),
		Category("Window Layout"),
		Description("Gets or sets the left position, in pixels. This value is relative to the upper-left corner of the screen."),
		DefaultValue("")
		]
		public Unit Left {
			get {
				object savedState;
                
				savedState = this.ViewState["Left"];
				if (savedState != null) {
					return (Unit) savedState;
				}
				return Unit.Empty;
			}
			set { 
				ViewState["Left"] = value; 
			}
		}

		/// <summary>
		/// Specifies the height of the window, in pixels.
		/// </summary>
		[
		Bindable(true),
		Category("Window Layout"),
		Description("Gets or sets the height of the window, in pixels."),
		DefaultValue("")
		]
		public Unit WindowHeight {
			get {
				object savedState;
                
				savedState = this.ViewState["Height"];
				if (savedState != null) {
					return (Unit) savedState;
				}
				return Unit.Empty;
			}
			set { 
				ViewState["Height"] = value; 
			}
		}

		/// <summary>
		/// Gets or sets the width of the window, in pixels.
		/// </summary>
		[
		Bindable(true),
		Category("Window Layout"),
		Description("Gets or sets the width of the window, in pixels."),
		DefaultValue("")
		]
		public Unit WindowWidth {
			get {
				object savedState;
                
				savedState = this.ViewState["Width"];
				if (savedState != null) {
					return (Unit) savedState;
				}
				return Unit.Empty;
			}
			set { 
				ViewState["Width"] = value; 
			}
		}

		protected override void AddAttributesToRender( HtmlTextWriter writer ) {
			//HACK: Add our attributes before calling the base. This way, our onclick
			//event is written before any validator's onclick, if any. Then our
			//type attribute as "button" is written before base's type "submit".
			//In fact, this is not the proper way to do it and may have side effects.
			//A better way would be deriving the class from something else, not button.
			writer.AddAttribute(HtmlTextWriterAttribute.Onclick.ToString(), GetAttributes(), true );
			writer.AddAttribute(HtmlTextWriterAttribute.Type.ToString(), "button", true );
			base.AddAttributesToRender( writer );
		}
		
		internal String GetAttributes() 
		{
			StringBuilder Options = new StringBuilder();

			Options.Append ("javascript:window.open( '");
			Options.Append (this.ResolveUrl(NavigateUrl));
			Options.Append ("', '");
			Options.Append (WindowName);
			Options.Append ("', 'menubar=");
			
			// MenuBar
			if ( MenuBar ) 
			{
				Options.Append( "1," );
			}
			else 
			{
				Options.Append( "0," );
			}

			// ToolBar
			Options.Append ("toolbar=");
			if ( ToolBar ) 
			{
				Options.Append( "1," );
			}
			else 
			{
				Options.Append( "0," );
			}

			// Status
			Options.Append ("status=");
			if ( Status ) 
			{
				Options.Append( "1," );
			}
			else 
			{
				Options.Append( "0," );
			}

			// ScrollBar
			//
			// Old: ("scrollbar=")
			// Bug fixed. Thank you Max.
			Options.Append ("scrollbars=");
			if ( ScrollBars ) 
			{
				Options.Append( "1," );
			}
			else 
			{
				Options.Append( "0," );
			}
			
			// Location
			Options.Append ("location=");
			if ( Location ) 
			{
				Options.Append( "1," );
			}
			else 
			{
				Options.Append( "0," );
			}

			// Resizable
			Options.Append ("resizable=");
			if ( Resizable ) 
			{
				Options.Append( "1," );
			}
			else 
			{
				Options.Append( "0," );
			}

			if ( Top != Unit.Empty ) 
			{
				Options.Append ("top=");
				Options.Append ( Top.Value.ToString() );
				Options.Append (",");
			}
			
			if ( Left != Unit.Empty ) 
			{
				Options.Append ("left=");
				Options.Append ( Left.Value.ToString() );
				Options.Append (",");
			}	

			if ( WindowHeight != Unit.Empty ) 
			{
				Options.Append ("height=");
				Options.Append ( WindowHeight.Value.ToString() );
				Options.Append (",");
			}	

			if ( WindowWidth != Unit.Empty ) 
			{
				Options.Append ("width=");
				Options.Append ( WindowWidth.Value.ToString() );
			}	

			Options.Append( "'); return false;" );

			return Options.ToString();
		}


	}

}