﻿using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using Bluelaser.Utilities;

[TestFixture()]
public class CreditCardUtilityTests
{
    [Test()]
    public void IsValidNumberTypeTest()
    {
        //=========================================================================
        // CreditCardUtilityTests Change Log
        //
        // Date             Developer       Description
        // 30 Aug 2007      T. Anglin       Changed test credit card numbers in
        //                                  GetCardTypeFromNumber and PassesLuhnTest
        //                                  to match PayPal issued test numbers.
        //=========================================================================

        string numDash = "4111-1111-1111-1111";
        string numSpace = "4111 1111 1111 1111";
        string numNoSpace = "4111111111111111";
        string numBadSep = "4111.1111.1111.1111";
        string numBadLen = "4111-1111-1111-111";

        Assert.IsTrue(CreditCardUtility.IsValidNumber(numDash), "IsValidNumber should allow numbers with dashes");
        Assert.IsTrue(CreditCardUtility.IsValidNumber(numSpace), "IsValidNumber should allow numbers with spaces");
        Assert.IsTrue(CreditCardUtility.IsValidNumber(numNoSpace), "IsValidNumber should allow numbers with no spaces");
        Assert.IsFalse(CreditCardUtility.IsValidNumber(numBadLen), "IsValidNumber should not allow numbers with too few numbers");
        Assert.IsFalse(CreditCardUtility.IsValidNumber(numBadSep), "IsValidNumber should not allow numbers with dot separators");
    }

    [Test()]    
    public void IsValidNumberCardTypeTest()
    {
        string visaDash = "4111-1111-1111-1111";
        string visaSpace = "4111 1111 1111 1111";
        string visaNoSpace = "4111111111111111";
        string visaBad = "4111.1111.1111.1111";

        Assert.IsTrue(CreditCardUtility.IsValidNumber(visaDash, CreditCardTypeType.Visa), "IsValidNumber should allow Visa numbers with dashes");
        Assert.IsTrue(CreditCardUtility.IsValidNumber(visaSpace, CreditCardTypeType.Visa), "IsValidNumber should allow Visa numbers with spaces");
        Assert.IsTrue(CreditCardUtility.IsValidNumber(visaNoSpace, CreditCardTypeType.Visa), "IsValidNumber should allow Visa numbers with no spaces");
        Assert.IsFalse(CreditCardUtility.IsValidNumber(visaBad, CreditCardTypeType.Visa), "IsValidNumber should not allow Visa numbers with decimals");

        string amexDash = "3400-000000-00009";
        string amexSpace = "3400 000000 00009";
        string amexNoSpace = "340000000000009";
        string amexBad = "3400.000000.00009";

        Assert.IsTrue(CreditCardUtility.IsValidNumber(amexDash, CreditCardTypeType.Amex), "IsValidNumber should allow Amex numbers with dashes");
        Assert.IsTrue(CreditCardUtility.IsValidNumber(amexSpace, CreditCardTypeType.Amex), "IsValidNumber should allow Amex numbers with spaces");
        Assert.IsTrue(CreditCardUtility.IsValidNumber(amexNoSpace, CreditCardTypeType.Amex), "IsValidNumber should allow Amex numbers with no spaces");
        Assert.IsFalse(CreditCardUtility.IsValidNumber(amexBad, CreditCardTypeType.Amex), "IsValidNumber should not allow Amex numbers with decimals");

        string mcDash = "5500-0000-0000-0004";
        string mcSpace = "5500 0000 0000 0004";
        string mcNoSpace = "5500000000000004";
        string mcBad = "5500.0000.0000.0004";

        Assert.IsTrue(CreditCardUtility.IsValidNumber(mcDash, CreditCardTypeType.MasterCard), "IsValidNumber should allow MC numbers with dashes");
        Assert.IsTrue(CreditCardUtility.IsValidNumber(mcSpace, CreditCardTypeType.MasterCard), "IsValidNumber should allow MC numbers with spaces");
        Assert.IsTrue(CreditCardUtility.IsValidNumber(mcNoSpace, CreditCardTypeType.MasterCard), "IsValidNumber should allow MC numbers with no spaces");
        Assert.IsFalse(CreditCardUtility.IsValidNumber(mcBad, CreditCardTypeType.MasterCard), "IsValidNumber should not allow MC numbers with decimals");

        string discoverDash = "6011-0000-0000-0004";
        string discoverSpace = "6011 0000 0000 0004";
        string discoverNoSpace = "6011000000000004";
        string discoverBad = "6011.0000.0000.0004";

        Assert.IsTrue(CreditCardUtility.IsValidNumber(discoverDash, CreditCardTypeType.Discover), "IsValidNumber should allow Discover numbers with dashes");
        Assert.IsTrue(CreditCardUtility.IsValidNumber(discoverSpace, CreditCardTypeType.Discover), "IsValidNumber should allow Discover numbers with spaces");
        Assert.IsTrue(CreditCardUtility.IsValidNumber(discoverNoSpace, CreditCardTypeType.Discover), "IsValidNumber should allow Discover numbers with no spaces");
        Assert.IsFalse(CreditCardUtility.IsValidNumber(discoverBad, CreditCardTypeType.Discover), "IsValidNumber should not allow Discover numbers with decimals");
    }

    [Test()]
    public void GetCardTypeFromNumberTest()
    {
        string visaNum = "4111-1111-1111-1111";
        string amexNum = "3782-822463-10005";
        string mcNum = "5105-1051-0510-5100";
        string discoverNum = "6011-1111-1111-1117";

        Assert.AreEqual(CreditCardTypeType.Visa, CreditCardUtility.GetCardTypeFromNumber(visaNum), "GetCardTypeFromNumber should return Visa");
        Assert.AreEqual(CreditCardTypeType.Amex, CreditCardUtility.GetCardTypeFromNumber(amexNum), "GetCardTypeFromNumber should return Amex");
        Assert.AreEqual(CreditCardTypeType.MasterCard, CreditCardUtility.GetCardTypeFromNumber(mcNum), "GetCardTypeFromNumber should return MasterCard");
        Assert.AreEqual(CreditCardTypeType.Discover, CreditCardUtility.GetCardTypeFromNumber(discoverNum), "GetCardTypeFromNumber should return Discover");
    }

    [Test()]
    public void PassesLuhnTestTest()
    {
        //According to PayPal, the valid test numbers that should be used
        //for testing card transactions are:
        //Credit Card Type              Credit Card Number
        //American Express              378282246310005
        //American Express              371449635398431
        //American Express Corporate    378734493671000
        //Diners Club                   30569309025904
        //Diners Club                   38520000023237
        //Discover                      6011111111111117
        //Discover                      6011000990139424
        //MasterCard                    5555555555554444
        //MasterCard                    5105105105105100
        //Visa                          4111111111111111
        //Visa                          4012888888881881
        //Src: https://www.paypal.com/en_US/vhelp/paypalmanager_help/credit_card_numbers.htm
        //Credit: Scott Dorman, http://www.geekswithblogs.net/sdorman

        string visaNum = "4111-1111-1111-1111";
        string amexNum = "3782-822463-10005";
        string mcNum = "5105-1051-0510-5100";
        string discoverNum = "6011-1111-1111-1117";

        Assert.IsTrue(CreditCardUtility.PassesLuhnTest(visaNum), "PassesLuhnTest should return true for this Visa number");
        Assert.IsTrue(CreditCardUtility.PassesLuhnTest(amexNum), "PassesLuhnTest should return true for this Amex number");
        Assert.IsTrue(CreditCardUtility.PassesLuhnTest(mcNum), "PassesLuhnTest should return true for this MC number");
        Assert.IsTrue(CreditCardUtility.PassesLuhnTest(discoverNum), "PassesLuhnTest should return true for this Discover number");

        string visaBad = "4111-0000-0000-0000";

        Assert.IsFalse(CreditCardUtility.PassesLuhnTest(visaBad), "PassesLuhnTest should return false for this number");
    }
}
