using System; 
using System.Data; 
using Northwind.DataLayer; 
using Northwind.BusinessObject; 
using System.Web.Script.Serialization; 
 
namespace Northwind.BusinessObject.Base 
{ 
     /// <summary>
     /// Base class for Products.  Do not make changes to this class,
     /// instead, put additional code in the Products class 
     /// </summary>
     public class ProductsBase
     { 
         /// <summary> 
         /// Gets or Sets ProductID 
         /// </summary> 
         public int ProductID { get; set; } 
 
         /// <summary> 
         /// Gets or Sets ProductName 
         /// </summary> 
         public string ProductName { get; set; } 
 
         /// <summary> 
         /// Gets or Sets SupplierID 
         /// </summary> 
         public int? SupplierID { get; set; } 
 
         /// <summary> 
         /// Gets or Sets CategoryID 
         /// </summary> 
         public int? CategoryID { get; set; } 
 
         /// <summary> 
         /// Gets or Sets QuantityPerUnit 
         /// </summary> 
         public string QuantityPerUnit { get; set; } 
 
         /// <summary> 
         /// Gets or Sets UnitPrice 
         /// </summary> 
         public decimal? UnitPrice { get; set; } 
 
         /// <summary> 
         /// Gets or Sets UnitsInStock 
         /// </summary> 
         public Int16? UnitsInStock { get; set; } 
 
         /// <summary> 
         /// Gets or Sets UnitsOnOrder 
         /// </summary> 
         public Int16? UnitsOnOrder { get; set; } 
 
         /// <summary> 
         /// Gets or Sets ReorderLevel 
         /// </summary> 
         public Int16? ReorderLevel { get; set; } 
 
         /// <summary> 
         /// Gets or Sets Discontinued 
         /// </summary> 
         public bool Discontinued { get; set; } 
 
         /// <summary> 
         /// Gets or sets the Related Suppliers.  Related to column SupplierID 
         /// </summary> 
         [ScriptIgnore] 
         public Lazy<Suppliers> Suppliers 
         { 
             get 
             { 
                 int value; 
                 bool hasValue = Int32.TryParse(SupplierID.ToString(), out value); 
 
                 if (hasValue) 
                     return new Lazy<Suppliers>(() => Northwind.BusinessObject.Suppliers.SelectByPrimaryKey(value)); 
                 else 
                     return null; 
             } 
             set{ } 
         }  
 
         /// <summary> 
         /// Gets or sets the Related Categories.  Related to column CategoryID 
         /// </summary> 
         [ScriptIgnore] 
         public Lazy<Categories> Categories 
         { 
             get 
             { 
                 int value; 
                 bool hasValue = Int32.TryParse(CategoryID.ToString(), out value); 
 
                 if (hasValue) 
                     return new Lazy<Categories>(() => Northwind.BusinessObject.Categories.SelectByPrimaryKey(value)); 
                 else 
                     return null; 
             } 
             set{ } 
         }  
 
 
         /// <summary> 
         /// Constructor 
         /// </summary> 
         public ProductsBase() 
         { 
         } 
 
         /// <summary>
         /// Selects a record by primary key(s) 
         /// </summary>
         public static Products SelectByPrimaryKey(int productID) 
         { 
             return ProductsDataLayer.SelectByPrimaryKey(productID); 
         } 
 
         /// <summary> 
         /// Selects all records as a collection (List) of Products 
         /// </summary> 
         public static ProductsCollection SelectAll() 
         { 
             return ProductsDataLayer.SelectAll(); 
         } 
 
         /// <summary> 
         /// Selects all records as a collection (List) of Products sorted by the sort expression 
         /// </summary> 
         public static ProductsCollection SelectAll(string sortExpression) 
         { 
             ProductsCollection objProductsCol = ProductsDataLayer.SelectAll();
             return SortByExpression(objProductsCol, sortExpression);
         } 
 
         /// <summary>
         /// Selects all Products by Suppliers, related to column SupplierID 
         /// </summary> 
         public static ProductsCollection SelectProductsCollectionBySuppliers(int supplierID) 
         { 
             return ProductsDataLayer.SelectProductsCollectionBySuppliers(supplierID); 
         } 
 
         /// <summary>
         /// Selects all Products by Suppliers, related to column SupplierID, sorted by the sort expression 
         /// </summary> 
         public static ProductsCollection SelectProductsCollectionBySuppliers(int supplierID, string sortExpression) 
         { 
             ProductsCollection objProductsCol = ProductsDataLayer.SelectProductsCollectionBySuppliers(supplierID); 
             return SortByExpression(objProductsCol, sortExpression);
         } 
 
         /// <summary>
         /// Selects all Products by Categories, related to column CategoryID 
         /// </summary> 
         public static ProductsCollection SelectProductsCollectionByCategories(int categoryID) 
         { 
             return ProductsDataLayer.SelectProductsCollectionByCategories(categoryID); 
         } 
 
         /// <summary>
         /// Selects all Products by Categories, related to column CategoryID, sorted by the sort expression 
         /// </summary> 
         public static ProductsCollection SelectProductsCollectionByCategories(int categoryID, string sortExpression) 
         { 
             ProductsCollection objProductsCol = ProductsDataLayer.SelectProductsCollectionByCategories(categoryID); 
             return SortByExpression(objProductsCol, sortExpression);
         } 
 
         /// <summary>
         /// Selects ProductID and ProductName columns for use with a DropDownList web control, ComboBox, CheckedBoxList, ListView, ListBox, etc 
         /// </summary> 
         public static ProductsCollection SelectProductsDropDownListData() 
         { 
             return ProductsDataLayer.SelectProductsDropDownListData(); 
         } 
 
         /// <summary>
         /// Sorts the ProductsCollection by sort expression 
         /// </summary> 
         public static ProductsCollection SortByExpression(ProductsCollection objProductsCol, string sortExpression) 
         { 
             bool isSortDescending = sortExpression.Contains(" DESC");

             if (isSortDescending)
                 sortExpression = sortExpression.Replace(" DESC", "");

             switch (sortExpression)
             {
                 case "ProductID":
                     objProductsCol.Sort(Northwind.BusinessObject.Products.ByProductID);
                     break;
                 case "ProductName":
                     objProductsCol.Sort(Northwind.BusinessObject.Products.ByProductName);
                     break;
                 case "SupplierID":
                     objProductsCol.Sort(Northwind.BusinessObject.Products.BySupplierID);
                     break;
                 case "CategoryID":
                     objProductsCol.Sort(Northwind.BusinessObject.Products.ByCategoryID);
                     break;
                 case "QuantityPerUnit":
                     objProductsCol.Sort(Northwind.BusinessObject.Products.ByQuantityPerUnit);
                     break;
                 case "UnitPrice":
                     objProductsCol.Sort(Northwind.BusinessObject.Products.ByUnitPrice);
                     break;
                 case "UnitsInStock":
                     objProductsCol.Sort(Northwind.BusinessObject.Products.ByUnitsInStock);
                     break;
                 case "UnitsOnOrder":
                     objProductsCol.Sort(Northwind.BusinessObject.Products.ByUnitsOnOrder);
                     break;
                 case "ReorderLevel":
                     objProductsCol.Sort(Northwind.BusinessObject.Products.ByReorderLevel);
                     break;
                 case "Discontinued":
                     objProductsCol.Sort(Northwind.BusinessObject.Products.ByDiscontinued);
                     break;
                 default:
                     break;
             }

             if (isSortDescending) 
                 objProductsCol.Reverse();

             return objProductsCol;
         } 
 
         /// <summary>
         /// Inserts a record 
         /// </summary> 
         public int Insert() 
         { 
             Products objProducts = (Products)this; 
             return ProductsDataLayer.Insert(objProducts); 
         } 
 
         /// <summary>
         /// Updates a record 
         /// </summary> 
         public void Update() 
         { 
             Products objProducts = (Products)this; 
             ProductsDataLayer.Update(objProducts); 
         } 
 
         /// <summary>
         /// Deletes a record based on primary key(s) 
         /// </summary>
         public static void Delete(int productID) 
         { 
             ProductsDataLayer.Delete(productID); 
         } 
 
         /// <summary> 
         /// Compares ProductID used for sorting 
         /// </summary> 
         public static Comparison<Products> ByProductID = delegate(Products x, Products y) 
         { 
             return x.ProductID.CompareTo(y.ProductID); 
         }; 
 
         /// <summary> 
         /// Compares ProductName used for sorting 
         /// </summary> 
         public static Comparison<Products> ByProductName = delegate(Products x, Products y) 
         { 
             string value1 = x.ProductName ?? String.Empty; 
             string value2 = y.ProductName ?? String.Empty; 
             return value1.CompareTo(value2); 
         }; 
 
         /// <summary> 
         /// Compares SupplierID used for sorting 
         /// </summary> 
         public static Comparison<Products> BySupplierID = delegate(Products x, Products y) 
         { 
             return Nullable.Compare(x.SupplierID, y.SupplierID); 
         }; 
 
         /// <summary> 
         /// Compares CategoryID used for sorting 
         /// </summary> 
         public static Comparison<Products> ByCategoryID = delegate(Products x, Products y) 
         { 
             return Nullable.Compare(x.CategoryID, y.CategoryID); 
         }; 
 
         /// <summary> 
         /// Compares QuantityPerUnit used for sorting 
         /// </summary> 
         public static Comparison<Products> ByQuantityPerUnit = delegate(Products x, Products y) 
         { 
             string value1 = x.QuantityPerUnit ?? String.Empty; 
             string value2 = y.QuantityPerUnit ?? String.Empty; 
             return value1.CompareTo(value2); 
         }; 
 
         /// <summary> 
         /// Compares UnitPrice used for sorting 
         /// </summary> 
         public static Comparison<Products> ByUnitPrice = delegate(Products x, Products y) 
         { 
             return Nullable.Compare(x.UnitPrice, y.UnitPrice); 
         }; 
 
         /// <summary> 
         /// Compares UnitsInStock used for sorting 
         /// </summary> 
         public static Comparison<Products> ByUnitsInStock = delegate(Products x, Products y) 
         { 
             return Nullable.Compare(x.UnitsInStock, y.UnitsInStock); 
         }; 
 
         /// <summary> 
         /// Compares UnitsOnOrder used for sorting 
         /// </summary> 
         public static Comparison<Products> ByUnitsOnOrder = delegate(Products x, Products y) 
         { 
             return Nullable.Compare(x.UnitsOnOrder, y.UnitsOnOrder); 
         }; 
 
         /// <summary> 
         /// Compares ReorderLevel used for sorting 
         /// </summary> 
         public static Comparison<Products> ByReorderLevel = delegate(Products x, Products y) 
         { 
             return Nullable.Compare(x.ReorderLevel, y.ReorderLevel); 
         }; 
 
         /// <summary> 
         /// Compares Discontinued used for sorting 
         /// </summary> 
         public static Comparison<Products> ByDiscontinued = delegate(Products x, Products y) 
         { 
             return x.Discontinued.CompareTo(y.Discontinued); 
         }; 
 
     } 
} 
