﻿using System;
using System.Web;
using System.Web.UI;

[assembly: WebResource("JavaScriptLibrary.JavaScript.ShowMessage.js", "application/x-javascript")]
[assembly: WebResource("JavaScriptLibrary.JavaScript.GreetUser.js", "application/x-javascript")]
[assembly: WebResource("JavaScriptLibrary.JavaScript.DummyFile.js", "application/x-javascript")]

namespace JavaScriptLibrary
{

    /// <summary>
    /// Helps include JavaScript files in pages.
    /// </summary>
    public class JavaScriptHelper
    {

        #region Constants

        private const string TEMPLATE_SCRIPT = "<script type=\"text/javascript\" src=\"{0}\"></script>\r\n";
        private const string NAME_SHOW_MESSAGE = "JavaScriptLibrary.JavaScript.ShowMessage.js";
        private const string NAME_GREET_USER = "JavaScriptLibrary.JavaScript.GreetUser.js";
        private const string NAME_DUMMY_FILE = "JavaScriptLibrary.JavaScript.DummyFile.js";
        private const string NAME_JQUERY = "jQuery";
        private const string URL_JQUERY = "http://ajax.aspnetcdn.com/ajax/jQuery/jquery-1.5.1.js";
        private const string URL_JQUERY_HTTPS = "https://ajax.aspnetcdn.com/ajax/jQuery/jquery-1.5.1.js";

        #endregion

        #region Public Methods

        /// <summary>
        /// Includes ShowMessage.js in the page.
        /// </summary>
        /// <param name="manager">Accessible via Page.ClientScript.</param>
        /// <param name="late">Include the JavaScript at the bottom of the HTML?</param>
        public static void Include_ShowMessage(ClientScriptManager manager, bool late = false)
        {
            IncludeJavaScript(manager, NAME_SHOW_MESSAGE, late);
        }

        /// <summary>
        /// Includes GreetUser.js (and all dependencies) in the page.
        /// </summary>
        /// <param name="manager">Accessible via Page.ClientScript.</param>
        /// <param name="late">Include the JavaScript at the bottom of the HTML?</param>
        public static void Include_GreetUser(ClientScriptManager manager, bool late = false)
        {

            // Dependency (ShowMessage.js).
            Include_ShowMessage(manager, late);

            // Include GreetUser.js.
            IncludeJavaScript(manager, NAME_GREET_USER, late);

        }

        /// <summary>
        /// Includes jQuery.js in the page.
        /// </summary>
        /// <param name="page">The current page. Pass in null to get the current page from the HTTP context.</param>
        /// <param name="late">Include the JavaScript at the bottom of the HTML?</param>
        public static void Include_jQuery(Page page, bool late = false)
        {
            if (page == null)
            {
                page = (Page)HttpContext.Current.Handler;
            }
            IncludeExternalJavaScript(page, NAME_JQUERY, URL_JQUERY, URL_JQUERY_HTTPS, late);
        }

        /// <summary>
        /// Excludes jQuery.js from the page.
        /// </summary>
        /// <param name="manager">Accessible via Page.ClientScript.</param>
        public static void Exclude_jQuery(ClientScriptManager manager)
        {
            ExcludeJavaScript(manager, NAME_JQUERY);
        }

        #endregion

        #region Private Methods

        /// <summary>
        /// Includes the specified embedded JavaScript file in the page.
        /// </summary>
        /// <param name="manager">Accessible via Page.ClientScript.</param>
        /// <param name="resourceName">The name used to identify the embedded JavaScript file.</param>
        /// <param name="late">Include the JavaScript at the bottom of the HTML?</param>
        private static void IncludeJavaScript(ClientScriptManager manager, string resourceName, bool late)
        {
            var type = typeof(JavaScriptLibrary.JavaScriptHelper);
            if (!manager.IsStartupScriptRegistered(type, resourceName))
            {
                if (late)
                {
                    var url = manager.GetWebResourceUrl(type, resourceName);
                    var scriptBlock = string.Format(TEMPLATE_SCRIPT, HttpUtility.HtmlEncode(url));
                    manager.RegisterStartupScript(type, resourceName, scriptBlock);
                }
                else
                {
                    manager.RegisterClientScriptResource(type, resourceName);
                    manager.RegisterStartupScript(type, resourceName, string.Empty);
                }
            }
        }

        /// <summary>
        /// Includes the specified external JavaScript file in the page.
        /// </summary>
        /// <param name="page">The current page.</param>
        /// <param name="key">The name used to uniquely identify the external JavaScript file.</param>
        /// <param name="httpUrl">The URL of the external JavaScript file.</param>
        /// <param name="httpsUrl">The URL of the external JavaScript file when SSL is enabled.</param>
        /// <param name="late">Include the JavaScript at the bottom of the HTML?</param>
        private static void IncludeExternalJavaScript(Page page, string key, string httpUrl, string httpsUrl, bool late)
        {
            var manager = page.ClientScript;
            var type = typeof(JavaScriptLibrary.JavaScriptHelper);
            bool isStartupRegistered = manager.IsStartupScriptRegistered(type, key);
            bool isScriptRegistered = manager.IsClientScriptIncludeRegistered(type, key);
            if (!(isStartupRegistered || isScriptRegistered))
            {
                string url;
                if (page.Request.Url.Scheme.ToLower() == "http")
                {
                    url = httpUrl;
                }
                else
                {
                    url = httpsUrl;
                }
                if (late)
                {
                    manager.RegisterStartupScript(type, key, string.Format(TEMPLATE_SCRIPT, HttpUtility.HtmlEncode(url)));
                }
                else
                {
                    manager.RegisterClientScriptInclude(type, key, url);
                }
            }
        }

        /// <summary>
        /// Registers a dummy script to prevent the inclusion of the real JavaScript.
        /// </summary>
        /// <param name="manager">Accessible via Page.ClientScript.</param>
        /// <param name="key">The name used to uniquely identify the JavaScript file.</param>
        private static void ExcludeJavaScript(ClientScriptManager manager, string key)
        {
            var type = typeof(JavaScriptLibrary.JavaScriptHelper);
            var url = manager.GetWebResourceUrl(type, NAME_DUMMY_FILE);
            manager.RegisterStartupScript(type, key, string.Empty);
            manager.RegisterClientScriptInclude(type, key, url);
        }

        #endregion

    }

}