﻿using System.Data;
using System.Data.SqlClient;
using System.Configuration;
using System.Linq;
using System.Collections.Generic;
using System;

namespace ZipCodeObjects
{
    /// <summary> 
    /// ZipCodeCollection 
    /// </summary> 
    /// <remarks> 
    /// Contains Collection of zip codes and meta. 
    /// </remarks> 
    public class ZipCodeCollection
    {

        /// <summary> 
        /// ReadLock 
        /// </summary> 
        /// <remarks> 
        /// Used By SyncLock to ensure Thread Safety 
        /// </remarks> 
        private static readonly object ReadLock = new object();

        /// <summary> 
        /// ZipCodeInstance 
        /// </summary> 
        /// <remarks> 
        /// ZipCodeInstance 
        /// Singleton Collection of Zip Codes and meta. 
        /// </remarks> 
        private static List<ZipCode> m_ZipCodeInstance;
        public static List<ZipCode> ZipCodeInstance
        {
            get
            {
                // initialize if not already done 
                if (m_ZipCodeInstance == null)
                {

                    //only allow 1 person to load data at once. 
                    lock (ReadLock)
                    {
                        if (m_ZipCodeInstance == null)
                        {
                            m_ZipCodeInstance = LoadData();
                        }
                    }

                }
                return m_ZipCodeInstance;
            }
        }

        /// <summary> 
        /// GetZipCodes 
        /// </summary> 
        /// <param name="ResultsPerPage"></param> 
        /// <param name="PageNumber"></param> 
        /// <returns> 
        /// IEnumerable(Of ZipCode) 
        /// </returns> 
        /// <remarks> 
        /// Page Clone of Instance Data Using Linq 
        /// </remarks> 
        public static IEnumerable<ZipCodeObjects.ZipCode> GetZipCodes(int ResultsPerPage, int PageNumber)
        {

            //number of records to skip 
            int skip = (PageNumber - 1) * ResultsPerPage;

            //number of results per page. 
            int take = ResultsPerPage;

            //execute Linq query for result set
            IEnumerable<ZipCodeObjects.ZipCode> result = (from zipcodes in ZipCodeInstance select zipcodes).Skip(skip).Take(ResultsPerPage);

            //return result 
            return result;
        }

        /// <summary> 
        /// SelectCount 
        /// </summary> 
        /// <returns> 
        /// Integer 
        /// </returns> 
        /// <remarks> 
        /// Returns total number of records in instance. 
        /// Uses Linq 
        /// </remarks> 
        public static int SelectCount()
        {
            return (from zipcodes in ZipCodeInstance select zipcodes).Count();
        }

        /// <summary> 
        /// LoadData 
        /// </summary> 
        /// <returns> 
        /// List(Of ZipCode) 
        /// </returns> 
        /// <remarks> 
        /// Load collection of zip codes from database. 
        /// </remarks> 
        private static List<ZipCode> LoadData()
        {

            //create new instance of zip code collection 
            List<ZipCode> ziplist = new List<ZipCode>();

            //setup database connection 
            System.Data.SqlClient.SqlConnection conn = new System.Data.SqlClient.SqlConnection();
            conn.ConnectionString = ConfigurationManager.ConnectionStrings["ProgrammersJournal"].ConnectionString;

            //open connection 
            conn.Open();

            try
            {

                //define sql command 
                SqlCommand cmd = new SqlCommand("pj_getallzipcodes", conn);

                //execute and loop through reader. 
                using (System.Data.SqlClient.SqlDataReader reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {

                        //add zip object to list. 
                        ziplist.Add(new ZipCode(reader.GetSqlString(0).ToString(), reader.GetSqlString(1).ToString(), reader.GetSqlString(2).ToString(), Convert.ToDecimal(reader.GetSqlString(3).ToString()), Convert.ToDecimal(reader.GetSqlString(4).ToString()), Convert.ToInt32(reader.GetSqlString(5).ToString()), Convert.ToInt16(reader.GetSqlString(6).ToString())));
                    }
                }
            }

            catch (Exception ex)
            {

                //bubble exception 
                throw new Exception(ex.Message);
            }

            finally
            {

                //close connection 
                if (conn.State != ConnectionState.Closed)
                {
                    conn.Close();
                }

                //instance is populated 
                //force garbage collection 
                GC.Collect();
                GC.WaitForPendingFinalizers();

            }

            //return new instance of zip code collection 
            return ziplist;
        }

        /// <summary> 
        /// New 
        /// </summary> 
        /// <remarks> 
        /// Conrtructor set to private to ensure 
        /// instances of the obect cannot be created 
        /// outside of this class. 
        /// </remarks> 
        private ZipCodeCollection()
        {

        }

    } 

    /// <summary> 
    /// ZipCode 
    /// </summary> 
    /// <remarks> 
    /// Single Zip code record and associated meta. 
    /// </remarks> 
    public class ZipCode
    {

        /// <summary> 
        /// Zip 
        /// </summary> 
        /// <remarks> 
        /// Zip Code 
        /// </remarks> 
        private string m_Zip;
        public string Zip
        {
            get { return m_Zip; }
        }

        /// <summary> 
        /// City 
        /// </summary> 
        /// <remarks> 
        /// City where zip code is located. 
        /// </remarks> 
        private string m_City;
        public string City
        {
            get { return m_City; }
        }

        /// <summary> 
        /// State 
        /// </summary> 
        /// <remarks> 
        /// State where zip code is located. 
        /// </remarks> 
        private string m_State;
        public string State
        {
            get { return m_State; }
        }

        /// <summary> 
        /// Latitude 
        /// </summary> 
        /// <remarks> 
        /// Latitude reference for this zip code. 
        /// </remarks> 
        private decimal m_Latitude;
        public decimal Latitude
        {
            get { return m_Latitude; }
        }

        /// <summary> 
        /// Longitude 
        /// </summary> 
        /// <remarks> 
        /// Longitude reference for this zip code. 
        /// </remarks> 
        private decimal m_Longitude;
        public decimal Longitude
        {
            get { return m_Longitude; }
        }

        /// <summary> 
        /// TimeZone 
        /// </summary> 
        /// <remarks> 
        /// TimeZone reference for this zip code. 
        /// </remarks> 
        private int m_TimeZone;
        public int TimeZone
        {
            get { return m_TimeZone; }
        }

        /// <summary> 
        /// Dst 
        /// </summary> 
        /// <remarks> 
        /// Dst reference for this zip code. 
        /// </remarks> 
        private short m_Dst;
        public short Dst
        {
            get { return m_Dst; }
        }

        /// <summary> 
        /// New 
        /// </summary> 
        /// <remarks> 
        /// parameterless constructor 
        /// </remarks> 
        public ZipCode()
        {

        }

        /// <summary> 
        /// New 
        /// </summary> 
        /// <param name="zip"></param> 
        /// <param name="city"></param> 
        /// <param name="state"></param> 
        /// <param name="latitude"></param> 
        /// <param name="longitude"></param> 
        /// <param name="timeZone"></param> 
        /// <param name="dst"></param> 
        /// <remarks> 
        /// Custom Contructor 
        /// </remarks> 
        public ZipCode(string zip, string city, string state, decimal latitude, decimal longitude, int timeZone, short dst)
        {

            this.m_Zip = zip;
            this.m_City = city;
            this.m_State = state;
            this.m_Latitude = latitude;
            this.m_Longitude = longitude;
            this.m_TimeZone = timeZone;
            this.m_Dst = dst;

        }

    } 
}
