﻿Imports System.Data
Imports System.Data.SqlClient
Imports System.Configuration
Imports System.Linq

Namespace ZipCodeObjects

    ''' <summary>
    ''' ZipCodeCollection
    ''' </summary>
    ''' <remarks>
    ''' Contains Collection of zip codes and meta.
    ''' </remarks>
    Public Class ZipCodeCollection

        ''' <summary>
        ''' ReadLock
        ''' </summary>
        ''' <remarks>
        ''' Used By SyncLock to ensure Thread Safety
        ''' </remarks>
        Private Shared ReadOnly ReadLock As New Object()

        ''' <summary>
        ''' ZipCodeInstance
        ''' </summary>
        ''' <remarks>
        ''' ZipCodeInstance
        ''' Singleton Collection of Zip Codes and meta.
        ''' </remarks>
        Private Shared m_ZipCodeInstance As List(Of ZipCode)
        Public Shared ReadOnly Property ZipCodeInstance() As List(Of ZipCode)
            Get
                ' initialize if not already done
                If m_ZipCodeInstance Is Nothing Then

                    'only allow 1 person to load data at once.
                    SyncLock ReadLock
                        If m_ZipCodeInstance Is Nothing Then
                            m_ZipCodeInstance = LoadData()
                        End If
                    End SyncLock

                End If
                Return m_ZipCodeInstance
            End Get
        End Property

        ''' <summary>
        ''' GetZipCodes
        ''' </summary>
        ''' <param name="ResultsPerPage"></param>
        ''' <param name="PageNumber"></param>
        ''' <returns>
        ''' IEnumerable(Of ZipCode)
        ''' </returns>
        ''' <remarks>
        ''' Page Clone of Instance Data Using Linq
        ''' </remarks>
        Public Shared Function GetZipCodes(ByVal ResultsPerPage As Integer, _
                                           ByVal PageNumber As Integer) As IEnumerable(Of ZipCode)

            'number of records to skip
            Dim skip As Integer = (PageNumber - 1) * ResultsPerPage

            'number of results per page.
            Dim take As Integer = ResultsPerPage

            'execute query for result set
            Dim result As IEnumerable(Of ZipCode) = _
                (From zipcodes In ZipCodeInstance).Skip(skip).Take(take)

            'return result
            Return result
        End Function

        ''' <summary>
        ''' SelectCount
        ''' </summary>
        ''' <returns>
        ''' Integer
        ''' </returns>
        ''' <remarks>
        ''' Returns total number of records in instance.
        ''' Uses Linq
        ''' </remarks>
        Public Shared Function SelectCount() As Integer
            Return (From zipcodes In ZipCodeInstance).Count()
        End Function

        ''' <summary>
        ''' LoadData
        ''' </summary>
        ''' <returns>
        ''' List(Of ZipCode)
        ''' </returns>
        ''' <remarks>
        ''' Load collection of zip codes from database.
        ''' </remarks>
        Private Shared Function LoadData() As List(Of ZipCode)

            'create new instance of zip code collection
            Dim ziplist As New List(Of ZipCode)

            'setup database connection
            Dim conn As New System.Data.SqlClient.SqlConnection()
            conn.ConnectionString = ConfigurationManager.ConnectionStrings("ProgrammersJournal").ConnectionString

            'open connection
            conn.Open()

            Try

                'define sql command
                Dim cmd As New SqlCommand("pj_getallzipcodes", conn)

                'execute and loop through reader.
                Using reader As SqlClient.SqlDataReader = cmd.ExecuteReader()
                    While reader.Read()

                        'add zip object to list.
                        ziplist.Add(New ZipCode(reader.GetSqlString(0).ToString(), _
                                                reader.GetSqlString(1).ToString(), _
                                                reader.GetSqlString(2).ToString(), _
                                                Convert.ToDecimal(reader.GetSqlString(3).ToString()), _
                                                Convert.ToDecimal(reader.GetSqlString(4).ToString()), _
                                                Convert.ToInt32(reader.GetSqlString(5).ToString()), _
                                                Convert.ToInt16(reader.GetSqlString(6).ToString())))
                    End While
                End Using

            Catch ex As Exception

                'bubble exception
                Throw New Exception(ex.Message)

            Finally

                'close connection
                If conn.State <> ConnectionState.Closed Then
                    conn.Close()
                End If

                'instance is populated
                'force garbage collection
                GC.Collect()
                GC.WaitForPendingFinalizers()

            End Try

            'return new instance of zip code collection
            Return ziplist
        End Function

        ''' <summary>
        ''' New
        ''' </summary>
        ''' <remarks>
        ''' Conrtructor set to private to ensure 
        ''' instances of the obect cannot be created
        ''' outside of this class.
        ''' </remarks>
        Private Sub New()

        End Sub

    End Class

    ''' <summary>
    ''' ZipCode
    ''' </summary>
    ''' <remarks>
    ''' Single Zip code record and associated meta.
    ''' </remarks>
    Public Class ZipCode

        ''' <summary>
        ''' Zip
        ''' </summary>
        ''' <remarks>
        ''' Zip Code
        ''' </remarks>
        Private m_Zip As String
        Public ReadOnly Property Zip() As String
            Get
                Return m_Zip
            End Get
        End Property

        ''' <summary>
        ''' City
        ''' </summary>
        ''' <remarks>
        ''' City where zip code is located.
        ''' </remarks>
        Private m_City As String
        Public ReadOnly Property City() As String
            Get
                Return m_City
            End Get
        End Property

        ''' <summary>
        ''' State
        ''' </summary>
        ''' <remarks>
        ''' State where zip code is located.
        ''' </remarks>
        Private m_State As String
        Public ReadOnly Property State() As String
            Get
                Return m_State
            End Get
        End Property

        ''' <summary>
        ''' Latitude
        ''' </summary>
        ''' <remarks>
        ''' Latitude reference for this zip code.
        ''' </remarks>
        Private m_Latitude As Decimal
        Public ReadOnly Property Latitude() As Decimal
            Get
                Return m_Latitude
            End Get
        End Property

        ''' <summary>
        ''' Longitude
        ''' </summary>
        ''' <remarks>
        ''' Longitude reference for this zip code.
        ''' </remarks>
        Private m_Longitude As Decimal
        Public ReadOnly Property Longitude() As Decimal
            Get
                Return m_Longitude
            End Get
        End Property

        ''' <summary>
        ''' TimeZone
        ''' </summary>
        ''' <remarks>
        ''' TimeZone reference for this zip code.
        ''' </remarks>
        Private m_TimeZone As Integer
        Public ReadOnly Property TimeZone() As Integer
            Get
                Return m_TimeZone
            End Get
        End Property

        ''' <summary>
        ''' Dst
        ''' </summary>
        ''' <remarks>
        ''' Dst reference for this zip code.
        ''' </remarks>
        Private m_Dst As Short
        Public ReadOnly Property Dst() As Short
            Get
                Return m_Dst
            End Get
        End Property

        ''' <summary>
        ''' New
        ''' </summary>
        ''' <remarks>
        ''' parameterless constructor
        ''' </remarks>
        Public Sub New()

        End Sub

        ''' <summary>
        ''' New
        ''' </summary>
        ''' <param name="zip"></param>
        ''' <param name="city"></param>
        ''' <param name="state"></param>
        ''' <param name="latitude"></param>
        ''' <param name="longitude"></param>
        ''' <param name="timeZone"></param>
        ''' <param name="dst"></param>
        ''' <remarks>
        ''' Custom Contructor
        ''' </remarks>
        Public Sub New(ByVal zip As String, _
                       ByVal city As String, _
                       ByVal state As String, _
                       ByVal latitude As Decimal, _
                       ByVal longitude As Decimal, _
                       ByVal timeZone As Integer, _
                       ByVal dst As Short)

            Me.m_Zip = zip
            Me.m_City = city
            Me.m_State = state
            Me.m_Latitude = latitude
            Me.m_Longitude = longitude
            Me.m_TimeZone = timeZone
            Me.m_Dst = dst

        End Sub

    End Class

End Namespace

