using System;
using System.ComponentModel;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel.Design;


namespace CustomGridFormatting
{
    [ProvideProperty("UseCustomReportFormat", typeof(GridView))]
    [ToolboxData("<{0}:CustomReportFormatExtender runat='server'></{0}:CustomReportFormatExtender>")]
    [ParseChildren(true, "Props")]
    [PersistChildren(false)]
    [Designer(typeof(CustomReportFormatExtenderDesigner))]
    public class CustomReportFormatExtender : Control, IExtenderProvider
    {

        #region Private fields
        private ExtenderPropertiesCollection _props = new ExtenderPropertiesCollection();
        #endregion

        #region IExtenderProvider support

        // IExtenderProvider requires us to identify which control types this extender applies to
        bool IExtenderProvider.CanExtend(object o)
        {
            return (o is GridView);
        }
        
        #endregion

        #region Visual Studio support
        // this function helps ensure compatibility with the Visual Studio designer
        protected void NotifyDesignerOfChange()
        {
            // Thanks to Paul Easter for this code on microsoft.public.dotnet.framework.aspnet.buildingcontrols

            // Tell the designer that the component has changed
            if (this.Site != null && this.Site.DesignMode)
            {
                try
                {
                    IDesignerHost host = (IDesignerHost)this.Site.GetService(typeof(IDesignerHost));
                    if (host != null)
                    {
                        IComponentChangeService changer = (IComponentChangeService)host.GetService(typeof(IComponentChangeService));
                        if (changer != null)
                        {
                            //changer.OnComponentChanging(this, null);
                            changer.OnComponentChanged(this, null, null, null);
                        }
                    }
                }
                catch
                {
                    // nothing for now;
                }
            }
        }

        #endregion

        #region Extender property support

        // we're exposing our collection of ExenderProperties here to ensure Visual Studio
        // will persist the extended property values for each GridView; without this,
        // extended property settings are not persisted
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
        [PersistenceMode(PersistenceMode.InnerProperty)]
        [Browsable(false)]
        public ExtenderPropertiesCollection Props
        {
            get
            {
                return _props;                
            }
        }


        // return the UseCustomReportFormat extended property value for the given control
        public bool GetUseCustomReportFormat(GridView grid)
        {
            return Props.GetUseCustomReportFormatForGridID(grid.ID);
        }

        // set the UseCustomReportFormat extended property value for the given control
        public void SetUseCustomReportFormat(GridView grid, bool value)
        {
            Props.SetUseCustomReportFormatForGridID(grid.ID, value);
            NotifyDesignerOfChange();  // ensure compatibility with Visual Studio
        }

        #endregion

        #region Extender functionality
        // Assign our custom report format code to the RowDataBound event for each
        // GridView that has UseCustomReportFormat set to true
        protected override void OnInit(EventArgs e)
        {
            base.OnInit(e);

            foreach (ExtenderProperties p in Props)
            {
                GridView g = FindControl(p.GridID) as GridView;
                if (g != null && p.UseCustomReportFormat)
                    g.RowDataBound += new GridViewRowEventHandler(Handle_RowDataBound);
            }            
        }

        // test a value of a cell; if it is not numeric, return -1
        private int GetCellValue(TableCell tc)
        {
            int i = -1;

            try { i = Convert.ToInt32(tc.Text); }
            catch { /* do nothing; */}

            return i;
        }

        // this is our custom report format code; this function is assigned 
        // as an event handler for a GridView's RowDataBound event
        protected void Handle_RowDataBound(object o, GridViewRowEventArgs e)
        {
            // apply custom formatting to data cells
            if (e.Row.RowType == DataControlRowType.DataRow)
            {
                // set formatting for the category cell
                TableCell cell = e.Row.Cells[0];
                cell.Width = new Unit("120px");
                cell.Style["border-right"] = "2px solid #666666";
                cell.BackColor = System.Drawing.Color.LightGray;

                // set formatting for value cells
                for (int i = 1; i < e.Row.Cells.Count; i++)
                {
                    cell = e.Row.Cells[i];

                    // right-align each of the column cells after the first
                    // and set the width
                    cell.HorizontalAlign = HorizontalAlign.Right;
                    cell.Width = new Unit("90px");

                    // alternate background colors
                    if (i % 2 == 1)
                        cell.BackColor
                          = System.Drawing.ColorTranslator.FromHtml("#EFEFEF");

                    // check value columns for a high enough value
                    // (value >= 8000) and apply special highlighting
                    if (GetCellValue(cell) >= 8000)
                    {
                        cell.Font.Bold = true;
                        cell.BorderWidth = new Unit("1px");
                        cell.BorderColor = System.Drawing.Color.Gray;
                        cell.BorderStyle = BorderStyle.Dotted;
                        cell.BackColor = System.Drawing.Color.Honeydew;
                    }

                }
            }

            // apply custom formatting to the header cells
            if (e.Row.RowType == DataControlRowType.Header)
            {
                foreach (TableCell cell in e.Row.Cells)
                {
                    cell.Style["border-bottom"] = "2px solid #666666";
                    cell.BackColor = System.Drawing.Color.LightGray;
                }
            }

        }

        #endregion

    }

}
