using System;
using System.Data;
using System.Configuration;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Web;
using System.Web.Security;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.WebControls.WebParts;
using System.Web.UI.HtmlControls;
using System.IO;
using ICSharpCode.SharpZipLib.Zip;
using Microsoft.Win32;

namespace Elsinore.Website
{
	public class ZipEntryHandler : IHttpHandler
	{
		private const string DEFAULT_CONTENT_TYPE = "text/plain";
		private const string ZIP_FILE_PARAM = "ZipFile";
		private const string ZIP_ENTRY_PARAM = "ZipEntry";

		#region Static Implementation

		private static string handlerPath;
		private static NameValueCollection values;

		static ZipEntryHandler()
		{
			handlerPath = Utility.GetHandlerPath(typeof(ZipEntryHandler));
			values = new NameValueCollection();
		}

		public static string GetUrl(string zipFileVirtualPath, string zipEntryName)
		{
			lock (values)
			{
				values[ZIP_FILE_PARAM] = zipFileVirtualPath;
				values[ZIP_ENTRY_PARAM] = zipEntryName;

				return Utility.GetUrl(handlerPath, values);
			}
		}

		public static ZipEntryInfo[] GetZipEntries(string zipFileVirtualPath, HttpContext context)
		{
			string filePath = context.Server.MapPath(zipFileVirtualPath);

			List<ZipEntryInfo> zipEntries = new List<ZipEntryInfo>();

			ZipFile zipFile = new ZipFile(filePath);

			try
			{
				foreach (ZipEntry zipEntry in zipFile)
				{
					if (!zipEntry.IsDirectory)
					{
						ZipEntryInfo zei = new ZipEntryInfo();
						zei.Url = ZipEntryHandler.GetUrl(zipFileVirtualPath, zipEntry.Name);
						zei.Name = zipEntry.Name;
						zei.Size = zipEntry.Size;
						zei.CompressedSize = zipEntry.CompressedSize;
						zei.DateTime = zipEntry.DateTime;
						zipEntries.Add(zei);
					}
				}
			}
			finally
			{
				zipFile.Close();
			}

			return zipEntries.ToArray();
		}

		#endregion

		public bool IsReusable
		{
			get { return true; }
		}

		public void ProcessRequest(HttpContext context)
		{
			string zipFileVirtualPath = context.Request.QueryString[ZIP_FILE_PARAM];
			string zipEntryName = context.Request.QueryString[ZIP_ENTRY_PARAM];

			string filePath = context.Server.MapPath(zipFileVirtualPath);


			ZipFile zipFile;

			try
			{
				zipFile = new ZipFile(filePath);
			}
			catch (FileNotFoundException)
			{
				string message = string.Format("The file '{0}' does not exist.", zipFileVirtualPath);
				throw new HttpException(404, message);
			}

			try
			{
				ZipEntry zipEntry = zipFile.GetEntry(zipEntryName);

				if (zipEntry == null)
				{
					string message = string.Format("The entry '{0}' does not exist in zip file '{1}'.", zipEntryName, zipFileVirtualPath);
					throw new HttpException(404, message);
				}

				context.Response.ContentType = ZipEntryHandler.GetContentType(zipEntryName);
				context.Response.AppendHeader("Content-Disposition", "filename=\"" + Path.GetFileName(zipEntryName) + "\"");

				using (Stream stream = zipFile.GetInputStream(zipEntry))
				{
					Utility.CopyStream(stream, context.Response.OutputStream);
				}
			}
			finally
			{
				zipFile.Close();
			}
		}

		public static string GetContentType(string filePath)
		{
			string extension = Path.GetExtension(filePath);

			if (!string.IsNullOrEmpty(extension))
			{
				using (RegistryKey key = Registry.ClassesRoot.OpenSubKey("MIME\\Database\\Content Type", false))
				{
					foreach (string keyName in key.GetSubKeyNames())
					{
						using (RegistryKey subKey = key.OpenSubKey(keyName, false))
						{
							string typeExtension = (string)subKey.GetValue("Extension");

							if (string.Equals(extension, typeExtension, StringComparison.CurrentCultureIgnoreCase))
							{
								return keyName;
							}
						}
					}
				}
			}

			return ZipEntryHandler.DEFAULT_CONTENT_TYPE;
		}
	}

	public class ZipEntryInfo
	{
		public string Url;
		public string Name;
		public long Size;
		public long CompressedSize;
		public DateTime DateTime;
	}
}
