﻿using System;
using System.Web;
using System.Reflection;
using System.Text;

namespace DataEncription.BusinessLogic
{
    /// <summary>
    /// Utility for encrypting and decrypting data.
    /// </summary>
    public static class Crypto
    {
        #region Fields
        private static bool _Initialized = false;
        private static MethodInfo _EncDecMethod = null;
        #endregion

        #region Methods
        /// <summary>
        /// Loads methods for encryption using reflection.
        /// </summary>
        /// <remarks>
        /// This method is executed only once, thus reducing
        /// the load of using reflection to a minimum.
        /// </remarks>
        public static void Initialize()
        {
            // Exit if methods are already initialized
            if (_Initialized) return;

            try
            {
                // This class is made public but the methods
                // we requiere are made private or internal.
                Type machineKeySection = typeof(System.Web.Configuration.MachineKeySection);

                // Get encryption / decryption method
                // You can look up all the method using Reflector
                // or a simillar tool.
                _EncDecMethod = machineKeySection.GetMethod("EncryptOrDecryptData",
                    BindingFlags.NonPublic | BindingFlags.Static, Type.DefaultBinder,
                    new[] { typeof(bool), typeof(byte[]), typeof(byte[]), typeof(int), typeof(int) },
                    null);

                // Set the init flag to true
                _Initialized = true;
            }
            catch
            {
                throw new Exception("Failed to initialized base encryption method.");
            }
        }

        /// <summary>
        /// Encrypts a string.
        /// </summary>
        /// <param name="data">Data to be encrypted.</param>
        /// <returns>Encrypted string.</returns>
        public static string Encrypt(string data)
        {
            // Make sure encryption method is loaded
            Initialize();

            // Try to encrypt data
            try
            {
                // Convert strig to byte array
                byte[] bytes = Encoding.UTF8.GetBytes(data);

                // Invoke the encryption method.
                // The method is basicaly the same for both
                // encryption and decription. You just have to
                // specify the the correct fEncrypt parameter
                // value.
                byte[] encData = (byte[])_EncDecMethod.Invoke(null, new object[] { true, bytes, null, 0, bytes.Length });

                // Return a re-usable string
                return Convert.ToBase64String(encData);
            }
            catch
            {
                // Data was not valid, encryption failed.
                throw new Exception("Failed to encrypt data.");
            }
        }

        /// <summary>
        /// Decrypts a string.
        /// </summary>
        /// <param name="data">Data to be decrypted.</param>
        /// <returns>Decrypted string.</returns>
        public static string Decrypt(string data)
        {
            // Make sure decryption method is loaded
            Initialize();

            // Try to decrypt data
            try
            {
                // Convert strig to byte array
                byte[] bytes = Convert.FromBase64String(data);

                // Invoke the decryption method.
                // The method is basicaly the same for both
                // encryption and decription. You just have to
                // specify the the correct fEncrypt parameter
                // value.
                byte[] decData = (byte[])_EncDecMethod.Invoke(null, new object[] { false, bytes, null, 0, bytes.Length });

                // Return a re-usable string
                return Encoding.UTF8.GetString(decData);
            }
            catch
            {
                // Data was not valid, encryption failed.
                throw new Exception("Failed to encrypt data.");
            }
        }
        #endregion
    }
}
